<?php
/**
 * @author      Flycart (Alagesan)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html
 * @link        https://www.flycart.org
 * */

namespace Wlpr\App\Models;
abstract class Base
{
    protected $table = NULL, $primary_key = NULL, $fields = array();
    static protected $db;
    function __construct()
    {
        global $wpdb;
        self::$db = $wpdb;
    }

    function getTableName()
    {
        return $this->table;
    }

    function getTablePrefix(){
        return self::$db->prefix;
    }

    function create()
    {
        if (is_multisite()) {
            // get ids of all sites
            $blog_table = self::$db->blogs;
            $blog_ids = self::$db->get_col("SELECT blog_id FROM {$blog_table}");
            foreach ($blog_ids as $blog_id) {
                switch_to_blog($blog_id);
            
                // create tables for each site
                $this->beforeTableCreation();
                $this->runTableCreation();
                $this->afterTableCreation();
                restore_current_blog();
            }
        } else {

            // activated on a single site
            $this->beforeTableCreation();
            $this->runTableCreation();
            $this->afterTableCreation();

        }
    }

    abstract function beforeTableCreation();

    abstract function runTableCreation();

    abstract function afterTableCreation();

    function saveData($data)
    {
        $primary_key = sanitize_key((isset($data[$this->primary_key]) && !empty($data[$this->primary_key])) ? $data[$this->primary_key] : 0);
        if (!empty($primary_key)) {
            $row = $this->getByKey($primary_key);
            if (!empty($row)) {
            } else {
                return $this->insertRow($data);
            }
            return $primary_key;
        } else {
            return $this->insertRow($data);
        }
    }

    function insertRow($data)
    {
        if (!empty($this->fields)) {
            $columns = implode('`,`', array_keys($this->fields));
            $values = implode(',', $this->fields);
            $actual_values = $this->formatData($data);
            $query = self::$db->prepare("INSERT INTO {$this->table} (`{$columns}`) VALUES ({$values});", $actual_values);
            self::$db->query($query);
            return self::$db->insert_id;
        }
        return 0;
    }

    function formatData($data)
    {
        $result = array();
        if (!empty($this->fields)) {
            foreach ($this->fields as $key => $value) {
                $key = trim($key);
                if ('%d' == trim($value)) {
                    $value = intval(isset($data[$key]) ? $data[$key] : 0);
                } elseif ('%f' == trim($value)) {
                    $value = floatval(isset($data[$key]) ? $data[$key] : 0);
                } else {
                    $value = isset($data[$key]) ? $data[$key] : NULL;
                    if (is_array($value) || is_object($value)) {
                        $value = json_encode($value);
                    }
                }
                $result[$key] = $value;
            }
        }
        return $result;
    }

    protected function createTable($query, $add_charset = true)
    {
        if ($add_charset) {
            $query = $query . ' ' . $this->getCollation() . ';';
        }
        $can_create_table = true;
        try {
            $tablefields = self::$db->get_results( "DESCRIBE {$this->table};" );
            if(!empty($tablefields)){
                $can_create_table = false;
            }
        }catch (\Exception $e){
        }
        if($can_create_table){
            self::$db->query($query);
        }
    }

    protected function getCollation() {
        $collate = '';

		if ( self::$db->has_cap( 'collation' ) ) {
			$collate = self::$db->get_charset_collate();
        }
        return $collate;
    }

    function getWhere($where, $select = '*', $single = true)
    {
        if (is_array($select) || is_object($select)) {
            $select = implode(',', $select);
        }
        if (empty($select)) {
            $select = '*';
        }
        $query = "SELECT {$select} FROM {$this->table} WHERE {$where};";
        if ($single) {
            return self::$db->get_row($query, OBJECT);
        } else {
            return self::$db->get_results($query, OBJECT);
        }
    }

    function rawQuery($query, $single = true)
    {
        if (empty($query)) {
            return '';
        }
        if ($single) {
            return self::$db->get_row($query, OBJECT);
        } else {
            return self::$db->get_results($query, OBJECT);
        }
    }

    function getAll($select = '*')
    {
        if (is_array($select) || is_object($select)) {
            $select = implode(',', $select);
        }
        if (empty($select)) {
            $select = '*';
        }
        $query = "SELECT {$select} FROM {$this->table};";
        return self::$db->get_results($query, OBJECT);
    }

    function updateRow($data, $where = array())
    {
        if (empty($data) || !is_array($data)) {
            return false;
        }
        return self::$db->update($this->table, $data, $where);
    }

    function deleteRow($where)
    {
        return self::$db->delete($this->table, $where);
    }

    function getByKey($key)
    {
        $key = sanitize_key($key);
        $query = self::$db->prepare("SELECT * FROM {$this->table} WHERE `{$this->primary_key}` = %d;", array($key));
        return self::$db->get_row($query, OBJECT);
    }

    function getPrimaryKey()
    {
        return $this->primary_key;
    }
}