<?php
/**
 * @author      Flycart (Alagesan)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html
 * @link        https://www.flycart.org
 * */

namespace Wlpr\App\Helpers;

use Wlpr\App\Models\Points;

class Referral extends Point
{
    public static $instance = null;

    /**
     * class instance
     * @param array $config
     * @return Referral|null
     * @since 1.0.0
     */
    public static function getInstance(array $config = array())
    {
        if (!self::$instance) {
            self::$instance = new self($config);
        }
        return self::$instance;
    }

    /**
     * check referral coupon before apply
     * @param $coupon_code
     * @return bool
     * @since 1.0.0
     */
    function checkReferCouponBeforeCouponApply($coupon_code)
    {
        global $wpdb;
        if (empty($coupon_code)) {
            return false;
        }
        $referral_settings = get_option('wlpr_referral');

        $wlpr_enable_referral = (isset($referral_settings['wlpr_enable_referral']) && !empty($referral_settings['wlpr_enable_referral'])) ? $referral_settings['wlpr_enable_referral'] : 'yes';
        if($wlpr_enable_referral === 'no'){
            return false;
        }
        $referral_discount_enable = (isset($referral_settings['wlpr_referral_discount']) && !empty($referral_settings['wlpr_referral_discount'])) ? $referral_settings['wlpr_referral_discount'] : 'no';
        if ($referral_discount_enable === 'no') {
            return false;
        }
        $current_email = $this->getCurrentEmail();
        $current_user = wp_get_current_user();
        if(isset($current_user->user_email) && !empty($current_user->user_email) && $current_user->user_email != $current_email ){
            return false;
        }
        $coupon_code = sanitize_text_field($coupon_code);
        $current_email = sanitize_email($current_email);
        if(!empty($current_email) && !empty($coupon_code)){
            $point = new Points();
            $where = $wpdb->prepare('user_email = %s AND refer_code = %s', array($current_email,$coupon_code));
            $point_data = $point->getWhere($where, '*', true);
            if(!empty($point_data)){
                return false;
            }
        }
        $referred_user = $this->isValidReferCoupon($coupon_code);
        if (!empty($referred_user)) {

            Loyalty::woocommerce()->set_refer_discount_code($coupon_code); // Set the coupon code in session
        }

        if (!empty($referred_user) && !empty($current_email)) {
            $referral_model = new \Wlpr\App\Models\Referral();
            $ref_where = $wpdb->prepare('new_user = %s AND status IN("%s","%s")', array($current_email, 'completed',''));
            $referral_data = $referral_model->getWhere($ref_where, '*', true);
            if (empty($referral_data)) {
                return true;
            }
        }
        return false;
    }

    /**
     * check is valid referral coupon
     * @param $coupon_code
     * @return array|object|string|void|null
     * @since 1.0.0
     */
    function isValidReferCoupon($coupon_code)
    {
        global $wpdb;
        $coupon_code = sanitize_text_field($coupon_code);
        if (empty($coupon_code)) return '';
        $point = new Points();
        $where = $wpdb->prepare('refer_code = %s', array($coupon_code));
        $point_data = $point->getWhere($where, '*', true);
        if (empty($point_data)) {
            return '';
        }
        return $point_data;
    }

    /**
     * Get Order Email
     * @return mixed|string|null
     * @since 1.0.0
     */
    function getCurrentEmail()
    {
        $woocommercer_helper = Loyalty::woocommerce();
        $input_helper = new Input();
        $postData = $input_helper->post_get('post_data','');
        $postDataArray = array();
        if (is_string($postData) && $postData != '') {
            parse_str($postData, $postDataArray);
        }
        $postBillingEmail = $input_helper->post_get('billing_email','');
        if(!empty($postBillingEmail)){
            $postBillingEmail = sanitize_email($postBillingEmail);
        }
        if ($postBillingEmail != '') {
            $postDataArray['billing_email'] = $postBillingEmail;
        }
        if (!get_current_user_id()) {
            $order_id = $input_helper->post_get('order-received',0);
            if(!empty($order_id)){
                $order_id = sanitize_key($order_id);
            }
            if ($order_id) {
                $order = $woocommercer_helper->getOrder($order_id);
                $postDataArray['billing_email'] = $order->get_billing_email();
            }
        }
        $user_email = '';
        if (isset($postDataArray['billing_email']) && $postDataArray['billing_email'] != '') {
            $user_email = $postDataArray['billing_email'];
        } else if ($user_id = get_current_user_id()) {
            $user_email = get_user_meta($user_id, 'billing_email', true);
            if ($user_email != '' && !empty($user_email)) {
                return $user_email;
            } else {
                $user_details = get_userdata($user_id);
                if (isset($user_details->data->user_email) && $user_details->data->user_email != '') {
                    $user_email = $user_details->data->user_email;
                    return $user_email;
                }
            }
        }
        if(empty($user_email) && isset(WC()->cart)){
            $customer = WC()->cart->get_customer();
            $user_email = $customer->get_billing_email();
        }
        return sanitize_email($user_email);
    }

    /**
     * add referral point
     * @param $referral_code
     * @param $order_email
     * @param $order_id
     * @since 1.0.0
     */
    function add_referral_points($referral_code, $order_email, $order_id)
    {
        global $wpdb;
        $referral_settings = get_option('wlpr_referral');
        $enable_referral = (isset($referral_settings['wlpr_enable_referral']) && !empty($referral_settings['wlpr_enable_referral'])) ? $referral_settings['wlpr_enable_referral'] : 'yes';
        $order_email = sanitize_email($order_email);
        //1. verify already referred
        if (!empty($referral_code) && !empty($order_email) && !empty($order_id) && $enable_referral == 'yes') {
            $referred_user = $this->isValidReferCoupon($referral_code);
            $referral_model = new \Wlpr\App\Models\Referral();
            // check already referral
            $ref_where = $wpdb->prepare('new_user = %s AND status IN ("%s","%s")', array($order_email,'completed',''));
            $check_already_available = $referral_model->getWhere($ref_where, '*', true);
            $login_user_email = wp_get_current_user()->user_email;
            /*
             * case:
             *  1. Login user email NOT EQUAL to referred user email (Removed)
             *  2. Order email NOT EQUAL to referred user email
             *  3. Order email NOT IN referral system(already earned via referral program)
             * */
            if (!empty($referred_user) && !empty($order_email) && !empty($referred_user->user_email) && $referred_user->user_email != $order_email && empty($check_already_available)) {
                $referral_model = new \Wlpr\App\Models\Referral();
                // check pending status
                $ref_where = $wpdb->prepare('new_user = %s AND status IN("%s")', array($order_email,'pending'));
                $referral_data = $referral_model->getWhere($ref_where, '*', true);
                if (empty($referral_data) || (isset($referral_data->status) && in_array($referral_data->status,array('pending'))) ) {
                    //2. check fixed point and increase point
                    $referred_by_earn_point = (isset($referral_settings['wlpr_referring_person']) && (!empty($referral_settings['wlpr_referring_person'])||$referral_settings['wlpr_referring_person'] ==0)) ? $referral_settings['wlpr_referring_person'] : 50;
                    $new_user_earn_point = (isset($referral_settings['wlpr_referred_person']) && (!empty($referral_settings['wlpr_referred_person'])||$referral_settings['wlpr_referred_person']==0)) ? $referral_settings['wlpr_referred_person'] : 50;
                    //3. check rule based point
                    $woocommerce_helper = Loyalty::woocommerce();
                    $order = $woocommerce_helper->getOrder($order_id);
                    // referral discount
                    $sub_total = 0;
                    $product_helper = Loyalty::product();
                    foreach ($order->get_items() as $item){
                        $product = $order->get_product_from_item($item);
                        $sub_total += $product_helper->getProductPrice($product) * $item['qty'];
                    }
                    //$sub_total = $order->get_subtotal();
                    $referral_rules = (isset($referral_settings['wlpr_rules']) && !empty($referral_settings['wlpr_rules'])) ? $referral_settings['wlpr_rules'] : array();
                    foreach ($referral_rules as $rule) {
                        if (isset($rule['referral_type']) && !empty($rule['referral_type']) && $rule['referral_type'] == 'referee') {
                            $new_user_earn_point += $this->get_rule_condition($rule, $sub_total);
                        } elseif (isset($rule['referral_type']) && !empty($rule['referral_type']) && $rule['referral_type'] == 'referrer') {
                            $referred_by_earn_point += $this->get_rule_condition($rule, $sub_total);
                        }
                    }
                    if ($referred_by_earn_point > 0) {
                        $this->addEarnPoint($referred_user->user_email, $referred_by_earn_point, 'referral_point', null, $order_id);
                    }
                    if ($new_user_earn_point > 0) {
                        $this->addEarnPoint($order_email, $new_user_earn_point, 'referring_user', null, $order_id);
                    }
                    $discount_amount = 0;
                    foreach ($order->get_items('coupon') as $coupon_item_id => $coupon_item) {
                        $coupon_code = $coupon_item->get_code();
                        if ($coupon_code == $referral_code) {
                            $discount_amount += $coupon_item->get_discount();
                        }
                    }
                    $data = array(
                        'referred_by' => sanitize_email($referred_user->user_email),
                        'new_user' => $order_email,
                        'referred_by_earn_point' => (int)$referred_by_earn_point,
                        'new_user_earn_point' => (int)$new_user_earn_point,
                        'referred_order_price' => $sub_total,
                        'new_user_discount' => $discount_amount,
                        'order_id' => $order_id,
                        'created_date' => date('Y-m-d H:i:s')
                    );
                    if(isset($referral_data->status) && in_array($referral_data->status,array('pending'))){
                        $update_data = array(
                            'referred_by' => sanitize_email($referred_user->user_email),
                            'new_user' => $order_email,
                            'referred_by_earn_point' => (int)$referred_by_earn_point,
                            'new_user_earn_point' => (int)$new_user_earn_point,
                            'referred_order_price' => $sub_total,
                            'new_user_discount' => $discount_amount,
                            'order_id' => $order_id,
                            'status' => 'completed'
                        );
                        $where = array('id' => (int)$referral_data->id);
                        $referral_model->updateRow($update_data, $where);
                    }else{
                        $data['status'] = 'completed';
                        $referral_model->insertRow($data);
                    }
                    \WC_Emails::instance();
                    do_action('wlpr_after_add_referral_points', $data, $order_id);
                }
            }
        }
    }

    function get_fixed_point($type = 'referee'){
        $referral_settings = get_option('wlpr_referral');
        //2. check fixed point and increase point
        $point = 0;
        if($type == 'referee'){
            $point += (isset($referral_settings['wlpr_referred_person']) && (!empty($referral_settings['wlpr_referred_person'])|| $referral_settings['wlpr_referred_person'] == 0)) ? $referral_settings['wlpr_referred_person'] : 50;
        }else{
            $point += (isset($referral_settings['wlpr_referring_person']) && (!empty($referral_settings['wlpr_referring_person']) || $referral_settings['wlpr_referring_person'] ==0 )) ? $referral_settings['wlpr_referring_person'] : 50;
        }
        return $point;
    }

    function get_rule_point($sub_total,$type = 'referee'){
        $referral_settings = get_option('wlpr_referral');
        $referral_rules = (isset($referral_settings['wlpr_rules']) && !empty($referral_settings['wlpr_rules'])) ? $referral_settings['wlpr_rules'] : array();
        $point = 0;
        foreach ($referral_rules as $rule) {
            if (isset($rule['referral_type']) && !empty($rule['referral_type']) && $rule['referral_type'] == 'referee' && $type == 'referee') {
                $point += $this->get_rule_condition($rule, $sub_total);
            } elseif (isset($rule['referral_type']) && !empty($rule['referral_type']) && $rule['referral_type'] == 'referrer'  && $type == 'referrer') {
                $point += $this->get_rule_condition($rule, $sub_total);
            }
        }
        return $point;
    }

    /**
     * Get rule condition process
     * @param $rule
     * @param $sub_total
     * @return float|int|mixed
     * @since 1.0.0
     */
    function get_rule_condition($rule, $sub_total)
    {
        $earn_point = 0;
        if (isset($rule['rule_min']) && isset($rule['rule_max']) && ($sub_total >= $rule['rule_min']) && ($sub_total <= $rule['rule_max'] || ($rule['rule_max'] == 0))) {
            if (isset($rule['earn_point']) && isset($rule['earn_point_type']) && $rule['earn_point_type'] == 'fixed') {
                $earn_point = $rule['earn_point'];
            } elseif (isset($rule['earn_point'])) {
                $earn_point = $sub_total * ($rule['earn_point'] / 100);
            }
        }
        //$rule_levels = $this->get_rule_levels($rule, $sub_total);
        /*foreach ($rule_levels as $level) {
            if (($sub_total >= $level['min_price']) && $sub_total <= $level['rule_max']) {
                $referred_by_earn_point += $level['earn_point'];
            }
        }*/
        return $earn_point;
    }

    /**
     * Get next referral rule level
     * @param $level
     * @param $interval
     * @param $sub_total
     * @return array
     * @since 1.0.0
     */
    function get_next_level_rule($level, $interval, $sub_total)
    {
        $next_level = array(
            'rule_min' => $level['rule_max'] + 1,
            'rule_max' => (($level['rule_max'] + 1) + $interval),
            'level_percentage' => $level['level_percentage'] + $level['level_percentage'],
        );
        if ($level['level_percentage_type'] != 'fixed') {
            $next_level['earn_point'] = ($level['level_percentage'] / 100) * $sub_total;
        } else {
            $next_level['earn_point'] = $level['level_percentage'];
        }
        return $next_level;
    }

    /**
     * Get referral rule levels
     * @param $rule
     * @param int $sub_total
     * @return array
     * @since 1.0.0
     */
    function get_rule_levels($rule, $sub_total = 0)
    {
        $first_level = array(
            'rule_min' => (isset($rule['rule_min']) && !empty($rule['rule_min'])) ? $rule['rule_min'] : 0,
            'rule_max' => (isset($rule['rule_max']) && !empty($rule['rule_max'])) ? $rule['rule_max'] : 0,
            'level_percentage' => (isset($rule['level_percentage']) && !empty($rule['level_percentage'])) ? $rule['level_percentage'] : 0,
            'level_percentage_type' => (isset($rule['level_percentage_type']) && !empty($rule['level_percentage_type'])) ? $rule['level_percentage_type'] : 'fixed',
            'earn_point' => ($rule['level_percentage_type'] != 'fixed') ? (($rule['level_percentage'] / 100) * $sub_total) : $rule['level_percentage']
        );
        $interval = ($rule['rule_max'] - $rule['rule_min']);
        $second_level = $this->get_next_level_rule($first_level, $interval, $sub_total);
        $thired_level = $this->get_next_level_rule($second_level, $interval, $sub_total);
        $four_level = $this->get_next_level_rule($thired_level, $interval, $sub_total);
        $five_level = $this->get_next_level_rule($four_level, $interval, $sub_total);
        return array($first_level, $second_level, $thired_level, $four_level, $five_level);
    }

    /**
     * validate referral rule before save
     * @param $data
     * @return mixed
     * @since 1.0.0
     */
    function validate_referral_rule($data)
    {
        if (isset($data['wlpr_rules']) && !empty($data['wlpr_rules'])) {
            $all_rule_levels = array();
            foreach ($data['wlpr_rules'] as $rule_key => $rule) {
                if (isset($rule['rule_type']) && $rule['rule_type'] == 'price') {
                    //validate price rule
                    $current_rule_levels = $this->get_rule_levels($rule);
                    $pass_rule = true;
                    foreach ($all_rule_levels as $all_rule_level) {
                        foreach ($current_rule_levels as $current_rule_level) {
                            if (($current_rule_level['min_price'] == $all_rule_level['min_price']) || ($current_rule_level['min_price'] <= $all_rule_level['rule_max'])) {
                                unset($data['wlpr_rules'][$rule_key]);
                                $pass_rule = false;
                                break 2;
                            }
                        }
                    }
                    if ($pass_rule) {
                        foreach ($current_rule_levels as $current_rule_level) {
                            $all_rule_levels[] = $current_rule_level;
                        }
                    }
                }
            }
        }
        return $data;
    }

    /**
     * Gat random code
     * @return bool|string
     */
    function get_random_code(){
        $permitted_chars = '0123456789abcdefghijklmnopqrstuvwxyz';
        return substr(str_shuffle($permitted_chars), 0, 9);
    }

    /**
     * Get unique refer code for user
     * @param string $ref_code
     * @param bool $recursive
     * @return string
     */
    function get_unique_refer_code($ref_code = '',$recursive = false){
        global $wpdb;
        $referral_settings = get_option('wlpr_referral');
        $prefix = (isset($referral_settings['wlpr_referral_prefix']) && !empty($referral_settings['wlpr_referral_prefix'])) ? $referral_settings['wlpr_referral_prefix'] : 'R_';
        $ref_code = !empty($ref_code) ? $ref_code: $prefix.$this->get_random_code();
        if(!empty($ref_code)){
            if($recursive){
                $ref_code = $prefix.$this->get_random_code();
            }
            $point = new Points();
            $ref_code = sanitize_text_field($ref_code);
            $where = $wpdb->prepare('refer_code=%s',array($ref_code));
            $user = $point->getWhere($where,'*',true);
            if(!empty($user)){
                return $this->get_unique_refer_code($ref_code,true);
            }
        }
        return $ref_code;
    }

    function getReferralUrl($code = '',$email = ''){
        global $wpdb;
        if(empty($code)){
            if(!empty($email)){
                $point_model = new Points();
                $email = sanitize_email($email);
                $where = $wpdb->prepare('user_email=%s',array($email));
                $point = $point_model->getWhere($where,'refer_code',true);
                if(!empty($point)){
                    $code = $point->refer_code;
                }
            }else{
                return '';
            }
        }
        if(empty($code)){
            return '';
        }
        return site_url() . '?wlpr_ref=' . $code;
    }

    function get_referral_detail($old_user,$new_user){
        global $wpdb;
        $referral_model = new \Wlpr\App\Models\Referral();
        $old_user = sanitize_email($old_user);
        $new_user = sanitize_email($new_user);
        $ref_where = $wpdb->prepare('referred_by = %s AND new_user = %s', array($old_user, $new_user));
        return $referral_model->getWhere($ref_where, '*', true);
    }

    function other_action_referral($event_type,$email)
    {
        global $wpdb;
        $referral_code = Loyalty::woocommerce()->get_refer_discount_code();
        if (!empty($referral_code) && in_array($event_type, array('account-signup', 'product-review'))) {
            $point = new Points();
            $referral_code = sanitize_text_field($referral_code);
            $where = $wpdb->prepare('refer_code=%s', array($referral_code));
            $user = $point->getWhere($where, '*', true);

            if (!empty($user) && isset($user->user_email) && !empty($user->user_email)) {
                $referral_data = $this->get_referral_detail($user->user_email, $email);
                if (empty($referral_data)) {

                    $data = array(
                        'referred_by' => sanitize_email($user->user_email),
                        'new_user' => sanitize_email($email),
                        'referred_by_earn_point' => 0,
                        'new_user_earn_point' => 0,
                        'referred_order_price' => 0.0,
                        'new_user_discount' => 0.0,
                        'order_id' => '',
                        'created_date' => date('Y-m-d H:s:i'),
                        'status' => 'pending',
                    );
                    try {
                        $referral_model = new \Wlpr\App\Models\Referral();
                        $referral_model->insertRow($data);
                        Loyalty::woocommerce()->set_refer_discount_code('');
                    }catch (\Exception $e){
                    }
                }
            }
        }
    }

    function check_referral_code($order_email){
        global $wpdb;
        $referral_code = '';
        $referral_model = new \Wlpr\App\Models\Referral();
        $order_email = sanitize_email($order_email);
        $ref_where = $wpdb->prepare('new_user = %s AND status = %s AND referred_by != %s', array($order_email,'pending',$order_email));
        $ref_another_case = $wpdb->prepare('(new_user=%s AND status = %s) || referred_by = %s',array($order_email,'completed',$order_email));
        $referral_data = $referral_model->getWhere($ref_where, '*', true);
        $ref_old_user_check = $referral_model->getWhere($ref_another_case,'*',true);

        if(!empty($referral_data) && isset($referral_data->referred_by) && $referral_data->referred_by){
            $point_user = new Points();
            $referral_data->referred_by = sanitize_email($referral_data->referred_by);
            $where = $wpdb->prepare('user_email = %s', array($referral_data->referred_by));
            $user = $point_user->getWhere($where, '*', true);
            if(!empty($user) && isset($user->refer_code) && !empty($user->refer_code)){
                $referral_code = $user->refer_code;
            }
        }elseif(!empty($ref_old_user_check)){
            $referral_code = '';
        }else{
            $referral_code = Loyalty::woocommerce()->get_refer_discount_code();
        }
        return $referral_code;
    }
}