<?php
/**
 * @author      Flycart (Alagesan)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html
 * @link        https://www.flycart.org
 * */

namespace Wlpr\App\Helpers;
class Check
{
    /**
     * Plugin activation check
     * @return bool
     */
    function pluginActivation()
    {
        $status = $this->init_check(true);
        return $status;
    }

    /**
     * initial check
     * @param bool $active_check
     * @return bool
     */
    function init_check($active_check = false)
    {
        $status = true;
        if (!$this->isEnvironmentCompatible()) {
            if ($active_check) {
                exit(WLPR_PLUGIN_NAME . __(' plugin can not be activated because it requires minimum PHP version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_PHP_VERSION);
            }
            $status = false;
        }
        if (!$this->isWordPressCompatible()) {
            if ($active_check) {
                exit(WLPR_PLUGIN_NAME . __(' plugin can not be activated because it requires minimum Wordpress version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_WP_VERSION);
            }
            $status = false;
        }
        if (!$this->isWoocommerceActive()) {
            if ($active_check) {
                exit(__('Woocommerce must installed and activated in-order to use ', WLPR_TEXT_DOMAIN) . WLPR_PLUGIN_NAME);
            }
            $status = false;
        }
        if (!$this->isWooCompatible()) {
            if ($active_check) {
                exit(WLPR_PLUGIN_NAME . __(' plugin can not be activated because it requires minimum Woocommerce version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_WC_VERSION);
            }
            $status = false;
        }
        return $status;
    }

    /**
     * Determines if the server environment is compatible with this plugin.
     *
     * @return bool
     * @since 1.0.0
     *
     */
    protected function isEnvironmentCompatible()
    {
        return version_compare(PHP_VERSION, WLPR_MINIMUM_PHP_VERSION, '>=');
    }

    /**
     * Determines if the WordPress compatible.
     *
     * @return bool
     * @since 1.0.0
     *
     */
    public function isWordPressCompatible()
    {
        if (!WLPR_MINIMUM_WP_VERSION) {
            $is_compatible = true;
        } else {
            $is_compatible = version_compare(get_bloginfo('version'), WLPR_MINIMUM_WP_VERSION, '>=');
        }
        return $is_compatible;
    }

    /**
     * Check the woocommerce is active or not
     * @return bool
     * @since 1.0.0
     */
    function isWoocommerceActive()
    {
        $active_plugins = apply_filters('active_plugins', get_option('active_plugins', array()));
        if (is_multisite()) {
            $active_plugins = array_merge($active_plugins, get_site_option('active_sitewide_plugins', array()));
        }
        return in_array('woocommerce/woocommerce.php', $active_plugins, false) || array_key_exists('woocommerce/woocommerce.php', $active_plugins);
    }

    /**
     * Woocommerce version
     * @return |null
     */
    function woo_version()
    {
        require_once ABSPATH . '/wp-admin/includes/plugin.php';
        $plugin_folder = get_plugins('/woocommerce');
        $plugin_file = 'woocommerce.php';
        $wc_installed_version = '1.0.0';
        if (isset($plugin_folder[$plugin_file]['Version'])) {
            $wc_installed_version = $plugin_folder[$plugin_file]['Version'];
        }
        return $wc_installed_version;
    }

    /**
     * Check woocommerce version is compatibility
     * @return bool
     * @since 1.0.0
     */
    function isWooCompatible()
    {
        $woo_version = $this->woo_version();
        if (!WLPR_MINIMUM_WC_VERSION) {
            $is_compatible = true;
        } else {
            $is_compatible = version_compare($woo_version, WLPR_MINIMUM_WC_VERSION, '>=');
        }
        return $is_compatible;
    }

    /**
     * inactive notice
     * @return string
     */
    function inactive_notice()
    {
        $message = '';
        if (!$this->isEnvironmentCompatible()) {
            $message = WLPR_PLUGIN_NAME . __(' is inactive. Because, it requires minimum PHP version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_PHP_VERSION;
        } elseif (!$this->isWordPressCompatible()) {
            $message = WLPR_PLUGIN_NAME . __(' is inactive. Because, it requires minimum Wordpress version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_WP_VERSION;
        } elseif (!$this->isWoocommerceActive()) {
            $message = __('Woocommerce must installed and activated in-order to use ', WLPR_TEXT_DOMAIN) . WLPR_PLUGIN_NAME;
        } elseif (!$this->isWooCompatible()) {
            $message = WLPR_PLUGIN_NAME . __(' is inactive. Because, it requires minimum Woocommerce version of ', WLPR_TEXT_DOMAIN) . WLPR_MINIMUM_WC_VERSION;
        }
        return '<div class="error"><p><strong>' . $message . '</strong></p></div>';
    }

    /**
     * Deactivates the plugin.
     *
     * @since 1.0.0
     */
    public function deactivatePlugin()
    {
        deactivate_plugins(WLPR_PLUGIN_FILE);
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }
    }
}