<?php
/**
 * @author      Flycart (Alagesan)
 * @license     http://www.gnu.org/licenses/gpl-3.0.html
 * @link        https://www.flycart.org
 * */

namespace Wlpr\App\Controllers\Admin;

use DateTime;
use DateTimeZone;
use Wlpr\App\Controllers\Base;
use Wlpr\App\Helpers\Check;
use Wlpr\App\Helpers\CsvHelper;
use Wlpr\App\Helpers\Launcher;
use Wlpr\App\Helpers\Pagination;
use Wlpr\App\Helpers\Validation;
use Wlpr\App\Helpers\Woocommerce;
use Wlpr\App\Models\PointAction;
use Wlpr\App\Models\PointLog;
use Wlpr\App\Models\Points;
use Exception;
use Wlpr\App\Helpers\Loyalty;
use Wlpr\App\Models\Referral;

class Main extends Base
{
    const MINIMUM_PHP_VERSION = '5.6.0';
    /** minimum WordPress version required by this plugin */
    const MINIMUM_WP_VERSION = '4.9';
    const MINIMUM_WC_VERSION = '3.0.9';

    /**
     * Init all the admin notice
     * @since 1.0.0
     */
    function initAdminNotices()
    {
        if (Woocommerce::hasAdminPrivilege()) {
            $plugin_check = new Check();
            if (!$plugin_check->isWordPressCompatible()) {
                $plugin_check->deactivatePlugin();
                $this->addAdminNotice('error', sprintf(
                    '%s requires WordPress version %s or higher. Please %supdate WordPress &raquo;%s',
                    '<strong>' . WLPR_PLUGIN_NAME . '</strong>',
                    self::MINIMUM_WP_VERSION,
                    '<a href="' . esc_url(admin_url('update-core.php')) . '">', '</a>'
                ));
            }
            if (!$plugin_check->isWooCompatible()) {
                $plugin_check->deactivatePlugin();
                $this->addAdminNotice('error', sprintf(
                    '%s requires WooCommerce version %s or higher. Please %supdate WooCommerce &raquo;%s',
                    '<strong>' . WLPR_PLUGIN_NAME . '</strong>',
                    self::MINIMUM_WC_VERSION,
                    '<a href="' . esc_url(admin_url('update-core.php')) . '">', '</a>'
                ));
            }
        }
    }

    /**
     * Initialize admin menu
     * @since 1.0.0
     */
    function addMenu()
    {
        if (Woocommerce::hasAdminPrivilege()) {
            global $submenu;
            if (isset($submenu['woocommerce'])) {
                add_submenu_page(
                    'woocommerce',
                    __('Loyalty Points and Rewards', WLPR_TEXT_DOMAIN),
                    __('Loyalty Points and Rewards', WLPR_TEXT_DOMAIN),
                    'manage_woocommerce',
                    WLPR_TEXT_DOMAIN,
                    array($this, 'managePages')
                );
            }
        }
    }

    /**
     * Add settings link
     * @param $links
     * @return array
     * @since 1.0.0
     */
    function pluginActionLinks($links)
    {
        if (!Woocommerce::hasAdminPrivilege()) {
            return $links;
        }
        $action_links = array(
            'dashboard' => '<a href="' . admin_url('admin.php?page=' . WLPR_PLUGIN_SLUG . '&view=dashboard') . '">' . __('Dashboard', WLPR_TEXT_DOMAIN) . '</a>',
            'settings' => '<a href="' . admin_url('admin.php?page=' . WLPR_PLUGIN_SLUG . '&view=settings') . '">' . __('Settings', WLPR_TEXT_DOMAIN) . '</a>',
            'customer' => '<a href="' . admin_url('admin.php?page=' . WLPR_PLUGIN_SLUG . '&view=customer') . '">' . __('Manage Points', WLPR_TEXT_DOMAIN) . '</a>',
            'referral' => '<a href="' . admin_url('admin.php?page=' . WLPR_PLUGIN_SLUG . '&view=referral') . '">' . __('Referrals', WLPR_TEXT_DOMAIN) . '</a>'
        );
        $return_links = array_merge($action_links, $links);
        return apply_filters('wlpr_point_action_links', $return_links);
    }

    /**
     * Add admin scripts
     * @param $hook - hook prefix
     * @since 1.0.0
     */
    function adminScripts($hook)
    {
        if (!Woocommerce::hasAdminPrivilege()) {
            return;
        }
        if (self::$input->get('page', NULL) != WLPR_PLUGIN_SLUG) {
            return;
        }
        $suffix = '.min';
        if(defined('SCRIPT_DEBUG')){
            $suffix  = SCRIPT_DEBUG ? '' : '.min';
        }
        // media library for launcher icon image
        wp_enqueue_media();
        //Register the scripts
        wp_register_script(WLPR_PLUGIN_SLUG . '-alertify', WLPR_PLUGIN_URL . 'Assets/Admin/Js/alertify'.$suffix.'.js', array(), WLPR_PLUGIN_VERSION);
        wp_register_script(WLPR_PLUGIN_SLUG . '-flatpickr-js', WLPR_PLUGIN_URL . 'Assets/Admin/Js/flatpickr.min.js', array(), WLPR_PLUGIN_VERSION);
        wp_register_script(WLPR_PLUGIN_SLUG . '-chart', WLPR_PLUGIN_URL . 'Assets/Admin/Js/gchart.min.js', array(), WLPR_PLUGIN_VERSION);
        wp_register_script(WLPR_PLUGIN_SLUG . '-select2', WLPR_PLUGIN_URL . 'Assets/Admin/Js/select2'.$suffix.'.js', array('jquery'), WLPR_PLUGIN_VERSION);
        //license key popup js
        wp_register_script(WLPR_PLUGIN_SLUG . '-sweetalert', WLPR_PLUGIN_URL . 'Assets/Admin/Js/sweetalert.min.js', array('jquery'), WLPR_PLUGIN_VERSION);
        wp_register_script(WLPR_PLUGIN_SLUG . '-main', WLPR_PLUGIN_URL . 'Assets/Admin/Js/wlpr-admin'.$suffix.'.js', array(), WLPR_PLUGIN_VERSION);

        //Enqueue the scripts
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-alertify');
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-flatpickr-js');
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-chart');
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-select2');
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-sweetalert');
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_script(WLPR_PLUGIN_SLUG . '-main');

        //Register the styles
        wp_register_style(WLPR_PLUGIN_SLUG . '-uptown', WLPR_PLUGIN_URL . 'Assets/Admin/Css/uptown.css', array(), WLPR_PLUGIN_VERSION);
        wp_register_style(WLPR_PLUGIN_SLUG . '-flatpickr', WLPR_PLUGIN_URL . 'Assets/Admin/Css/flatpickr'.$suffix.'.css', array(), WLPR_PLUGIN_VERSION);
        wp_register_style(WLPR_PLUGIN_SLUG . '-alertify', WLPR_PLUGIN_URL . 'Assets/Admin/Css/alertify'.$suffix.'.css', array(), WLPR_PLUGIN_VERSION);
        wp_register_style(WLPR_PLUGIN_SLUG . '-style-lpr', WLPR_PLUGIN_URL . 'Assets/Admin/Css/style-lpr.css', array(), WLPR_PLUGIN_VERSION);
        wp_register_style(WLPR_PLUGIN_SLUG . '-select2-css', WLPR_PLUGIN_URL . 'Assets/Admin/Css/select2'.$suffix.'.css', array(), WLPR_PLUGIN_VERSION);
        wp_register_style(WLPR_PLUGIN_SLUG . '-wlpr-admin-css', WLPR_PLUGIN_URL . 'Assets/Admin/Css/wlpr-admin'.$suffix.'.css', array(), WLPR_PLUGIN_VERSION);
        //Enqueue styles
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-uptown');
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-flatpickr');
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-alertify');
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-style-lpr');
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-select2-css');
        wp_enqueue_style(WLPR_PLUGIN_SLUG . '-wlpr-admin-css');
        wp_enqueue_style('wp-color-picker');
        //localize scripts
        $localize = array(
            'home_url' => get_home_url(),
            'admin_url' => admin_url(),
            'plugin_url' => WLPR_PLUGIN_URL,
            'user_email' => wp_get_current_user()->user_email,
            'customizer_content_width' => '',
            'ajax_url' => admin_url('admin-ajax.php'),
            'rule_types' => array(
                'price' => esc_html__('Cart subtotal', WLPR_TEXT_DOMAIN)
            ),
            'referral_types' => array(
                'referrer' => esc_html__('Advocate/Referrer', WLPR_TEXT_DOMAIN),
                'referee' => esc_html__('Friend', WLPR_TEXT_DOMAIN)
            ),
            'discount_types' => array(
                'fixed' => esc_html__('Fixed', WLPR_TEXT_DOMAIN),
                'percentage' => esc_html__('Percentage', WLPR_TEXT_DOMAIN)
            ),
            'min_text' => esc_html__('Min', WLPR_TEXT_DOMAIN),
            'max_text' => esc_html__('Max', WLPR_TEXT_DOMAIN),
            'remove_text' => esc_html__('Remove', WLPR_TEXT_DOMAIN),
            'localize_validating' => __('Validating...', WLPR_TEXT_DOMAIN),
            'localize_validate' => __('Validate  ', WLPR_TEXT_DOMAIN),
            'user_delete_title' => __('Remove User',WLPR_TEXT_DOMAIN),
            'user_delete_content' => __('Delete user from loyalty point system?',WLPR_TEXT_DOMAIN),
            'wlpr_apply_point_none' => Woocommerce::create_nonce('wlpr-apply-point-none'),
            'wlpr_delete_user_none' => Woocommerce::create_nonce('wlpr-delete-user-none'),
            'wlpr_private_note_none' => Woocommerce::create_nonce('wlpr-private-note-none'),
            'wlpr_update_expire_date_none' => Woocommerce::create_nonce('wlpr-update-expire-date-none'),
            'wlpr_popup_none' => Woocommerce::create_nonce('wlpr-popup-none'),
        );
        wp_localize_script(WLPR_PLUGIN_SLUG . '-main', 'wlpr_localize_data', $localize);
    }

    /**
     * Run on plugin activation
     * @return bool
     * @since 1.0.0
     */
    function pluginActivation()
    {
        $check = new Check();
        if ($check->init_check(true)) {
            try {
                $this->createRequiredTable();
            } catch (Exception $e) {
                exit(WLPR_PLUGIN_NAME . __('Plugin required table creation failed.', WLPR_TEXT_DOMAIN));
            }
        }
        return true;
    }

    /**
     * Create required table for plugin
    */
    public function createRequiredTable()
    {
        try {
            $point = new Points();
            $point->create();
            $point_log = new PointLog();
            $point_log->create();
            $point_expired = new PointAction();
            $point_expired->create();
            $referral = new Referral();
            $referral->create();
        } catch (Exception $e) {
            exit(WLPR_PLUGIN_NAME . __('Plugin required table creation failed.', WLPR_TEXT_DOMAIN));
        }
    }

    /**
     * Creating table whenever a new blog is created
     * @param $blog_id
     * @param $user_id
     * @param $domain
     * @param $path
     * @param $site_id
     * @param $meta
     */
    function onCreateBlog($blog_id, $user_id, $domain, $path, $site_id, $meta)
    {
        if (is_plugin_active_for_network(WLPR_PLUGIN_FILE)) {
            switch_to_blog($blog_id);
            $this->createRequiredTable();
            restore_current_blog();
        }
    }

    /**
     * When deleting the site delete the plugin related tables
     * @param $tables
     * @return array
     */
    function onDeleteBlog($tables)
    {
        $point = new Points();
        $tables[] = $point->getTableName();
        $point_log = new PointLog();
        $tables[] = $point_log->getTableName();
        $point_expired = new PointAction();
        $tables[] = $point_expired->getTableName();
        $referral = new Referral();
        $tables[] = $referral->getTableName();
        return $tables;
    }
    /**
     * process import data from csv file
    */
    function process_import()
    {
        $result = array();
        $wlpr_none = (string)self::$input->post_get('wlpr_none', '');
        if (Woocommerce::hasAdminPrivilege() && Woocommerce::verify_nonce($wlpr_none,'wlpr_import_preview_none')) {
            $csv_helper = Loyalty::csv();
            $table_params = array(
                'user_email' => '',
                'points' => '',
                'refer_code' => ''
            );
            $file_data = array();
            $limit_start = (int)self::$input->post_get('limit_start',0);
            $limit = (int)self::$input->post_get('limit', 5);
            $file_tmp_name = (string)self::$input->post_get('tmp_name', '');
            $original_file_name = (string)self::$input->post_get('name', '');
            $process = false;
            if(isset($original_file_name)){
                $ext = pathinfo($original_file_name, PATHINFO_EXTENSION);
                if(in_array($ext,array('csv'))){
                    $process = true;
                }
            }
            if ($process && isset($file_tmp_name) && !empty($file_tmp_name)) {

                $filename = basename(preg_replace('/[^a-zA-Z0-9\.\-\s+]/', '', html_entity_decode($original_file_name, ENT_QUOTES, 'UTF-8')));
                $file_path = WLPR_PLUGIN_PATH . 'App/File';
                $filepath = trim($file_path . '/' . $filename);
                $file_data = $csv_helper->getCsvData($filepath, $limit_start, $limit );
            }
            if (!empty($file_data)) {
                $need_update = (string)self::$input->post_get('need_update', 'no');
                $csv_helper->save_user($file_data, $need_update);
                $result['success'] = 'incomplete';
                $result['limit_start'] = $limit_start + $limit;
                $result['notification'] = sprintf(__('Insert/Update %s customer',WLPR_TEXT_DOMAIN),count($file_data));//__(sprintf('Insert/Update %s customer', count($file_data)), WLPR_TEXT_DOMAIN) . '<br>';
            }
            if (empty($result)) {
                if (isset($filepath) && !empty($filepath)) {
                    $csv_helper->file_delete($filepath);
                }
                $result['success'] = 'completed';
                $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer')));
            }
        }else{
            $result['success'] = 'error';
            $result['message'] = __('Invalid request', WLPR_TEXT_DOMAIN);
        }
        wp_send_json($result);
    }

    function export_popup(){
        $wlpr_none = (string)self::$input->post_get('wlpr_none','');
        if (!Woocommerce::hasAdminPrivilege() || !Woocommerce::verify_nonce($wlpr_none,'wlpr-popup-none')) {
            $result = array(
                'success' => 'failed'
            );
        }else{
            $path = WLPR_PLUGIN_PATH . 'App/File';
            $file_name = 'customer_export_*.*';
            $delete_file_path = trim($path.'/'.$file_name);
            foreach (glob($delete_file_path) as $file_path) {
                if (file_exists($file_path)) {
                    wp_delete_file($file_path);
                }
            }
            $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
            global $wpdb;
            $user_point_table = new Points();
            $table = $user_point_table->getTableName();
            $query = "SELECT count(*) as total_count FROM {$table} WHERE id > 0 ORDER BY id ASC";
            $point_user_count = $wpdb->get_row($query, OBJECT);
            $page_details = array(
                'base_url' => $base_url,
                'total_count' => $point_user_count->total_count,
                'process_count' => 0,
                'limit' => (int)1,
                'wlpr_none' => Woocommerce::create_nonce('wlpr-export-none')
            );
            $html = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/export_customer.php', $page_details)->render();
            $result = array(
                'success' => 'completed',
                'html' => $html
            );

        }
        wp_send_json($result);
    }

    function download_files(){
        $path = WLPR_PLUGIN_PATH . 'App/File';
        $file_name = 'customer_export_*.*';
        $delete_file_path = trim($path.'/'.$file_name);
        $download_list = array();
        foreach (glob($delete_file_path) as $file_path) {
            if (file_exists($file_path)) {
                $file_detail = new \stdClass();
                $file_detail->file_name = basename($file_path);
                $file_detail->file_path = $file_path;
                $download_list[] = $file_detail;
            }
        }
        return $download_list;
    }
    function export_files(){
        $wlpr_none = (string)self::$input->post_get('wlpr_none','');
        if (!Woocommerce::hasAdminPrivilege() || !Woocommerce::verify_nonce($wlpr_none,'wlpr-popup-none')) {
            $result = array(
                'success' => 'failed'
            );
        }else{

            $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));

            $page_details = array(
                'base_url' => $base_url,
                'files' => $this->download_files(),
                'wlpr_none' => Woocommerce::create_nonce('wlpr-export-none')
            );
            $html = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/export_download_files.php', $page_details)->render();
            $result = array(
                'success' => 'completed',
                'html' => $html
            );
        }
        wp_send_json($result);
    }

    function process_export(){
        $result = array();
        $wlpr_none = (string)self::$input->post_get('wlpr_none', '');
        if (Woocommerce::hasAdminPrivilege() && Woocommerce::verify_nonce($wlpr_none,'wlpr-export-none')) {
            $limit_start = (int)self::$input->post_get('limit_start',0);
            $limit = (int)self::$input->post_get('limit', 5);
            $total_count = (int)self::$input->post_get('total_count', 0);
            if ($total_count > $limit_start) {
                $path = WLPR_PLUGIN_PATH . 'App/File';
                $file_name = 'customer_export_';
                $file_count = 0;
                if ($limit_start >= 499) {
                    $file_count = round(($limit_start / 499));
                }
                $file_path = trim($path.'/'.$file_name.$file_count.'.csv');
                global $wpdb;
                $point_table = new Points();
                $table = $point_table->getTableName();
                $where = $wpdb->prepare('id > 0 ORDER BY id ASC LIMIT %d OFFSET %d;',array($limit,$limit_start));
                $query = "SELECT user_email,points,refer_code,birth_date FROM {$table} WHERE {$where}";
                $file_data = $wpdb->get_results($query, ARRAY_A);

                if(!empty($file_data)){
                    $csv_helper = new CsvHelper();
                    $csv_helper->setCsvData($file_path,$file_data);

                }
                $result['success'] = 'incomplete';
                $result['limit_start'] = $limit_start + $limit;
                if($result['limit_start'] >= $total_count){
                    $limit_start = $total_count;
                }
                $result['notification'] = sprintf(__('Exported %s customer',WLPR_TEXT_DOMAIN),$limit_start);//__(sprintf('Insert/Update %s customer', count($file_data)), WLPR_TEXT_DOMAIN) . '<br>';
            }else{
                $result['success'] = 'completed';
                $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer')));
            }
        }else{
            $result['success'] = 'error';
            $result['message'] = __('Invalid request', WLPR_TEXT_DOMAIN);
        }
        wp_send_json($result);
    }

    /**
     * Get import popup content
    */
    function import_popup()
    {
        $wlpr_none = (string)self::$input->post_get('wlpr_none','');
        if (!Woocommerce::hasAdminPrivilege() || !Woocommerce::verify_nonce($wlpr_none,'wlpr-popup-none')) {
            $result = array(
                'success' => 'failed'
            );
            wp_send_json($result);
        }else{
            $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_import_preview')));
            $page_details = array(
                'base_url' => $base_url,
                'wlpr_none' => Woocommerce::create_nonce('wlpr-import-none')
            );
            $html = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/import_customer.php', $page_details)->render();
            $result = array(
                'success' => 'completed',
                'html' => $html
            );
            wp_send_json($result);
        }
    }

    /**
     * Import popup preview content
    */
    function wlpr_import_popup_preview()
    {
        $result = array(
            'success' => 'failed',
            'html' => ''
        );

        $wplr_none = (string)self::$input->post_get('wlpr_none', '');
        if (Woocommerce::hasAdminPrivilege() && Woocommerce::verify_nonce($wplr_none,'wlpr-import-none') && isset($_FILES['importfile']) && !empty($_FILES['importfile'])) {
            $file = $_FILES['importfile'];
            $process = false;
            if(isset($file['name'])){
                $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                if(in_array($ext,array('csv'))){
                    $process = true;
                }
            }

            $redirect = false;
            if ((isset($file['error']) && $file['error'] > 0) ) {
                $redirect = true;
            }
            if (!empty($file['tmp_name']) && $process) {
                $delimiter  = (string)self::$input->post_get('delimiter', ',');
                $file_limit = (int)self::$input->post_get('limit', 10);
                $csv_helper = Loyalty::csv();
                $total = $csv_helper->getTotalRecord($file['tmp_name'], false);
                if ($total > 0) {
                    $header = $csv_helper->getFirstValue($file['tmp_name']);
                    $filename = basename(preg_replace('/[^a-zA-Z0-9\.\-\s+]/', '', html_entity_decode($file ['name'], ENT_QUOTES, 'UTF-8')));
                    $file_path = WLPR_PLUGIN_PATH . 'App/File';
                    $filepath = trim($file_path . '/' . $filename);

                    if (is_file($filepath)) {
                        if ($csv_helper->file_delete($filepath)) {
                            $csv_helper->file_upload($file ['tmp_name'], $filepath);
                            //$session->set('importfile',$file,'j2store');
                        }
                    } else {
                        $csv_helper->file_upload($file ['tmp_name'], $filepath);
                        //$session->set('importfile',$files,'j2store');
                    }
                    $need_update = (string)self::$input->post_get('need_update', 'no');
                    //$point_setting = get_option('wlpr_settings');
                    $page_details = array(
                        'need_update' => $need_update,
                        'total_count' => $total,
                        'process_count' => 0,
                        'header' => $header,
                        'file' => $file,
                        'base_url' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG))),
                        'action_url' => $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'process_import'))),
                        'limit' => $file_limit,
                        'wlpr_none' => Woocommerce::create_nonce('wlpr_import_preview_none')
                    );
                    $result['html'] = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/customer_import_preview.php', $page_details)->render();
                    $result['success'] = 'completed';
                } else {
                    $redirect = true;
                }
            } else {
                $redirect = true;
            }
            if ($redirect) {
                $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'import_customer')));
            }
        } else {
            $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'import_customer')));
        }
        wp_send_json($result);
    }

    /**
     * Migration
    */
    function process_migration(){
        $result = array();
        $allow_migrate = false;
        if (Woocommerce::hasAdminPrivilege()) {
            $allow_migrate = true;
        }
        $is_already_migration = get_option('wlpr_already_migration',0);

        $wlpr_none = (string)self::$input->post_get('wlpr_none', '');
        if($allow_migrate && Woocommerce::verify_nonce($wlpr_none,'wlpr-process-migration')){
            if($is_already_migration){
                $result['success'] = 'completed';
                $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
            }else{
                $limit_start = (int)self::$input->post_get('limit_start', 0);
                $limit = (int)self::$input->post_get('limit', 5);
                $total_count = (int)self::$input->post_get('total_count', 0);
                if($limit_start <= $total_count){
                    $this->doMigration($limit_start,$limit);
                    $result['success'] = 'incomplete';
                    $result['limit_start'] = $limit_start + $limit;
                    $result['notification'] = sprintf(__('%s Procesed Migration',WLPR_TEXT_DOMAIN),$limit);//__(sprintf('%s Procesed Migration', $limit), WLPR_TEXT_DOMAIN) . '<br>';
                }else{
                    update_option('wlpr_already_migration',1);
                    $result['success'] = 'completed';
                    $result['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
                }
            }
        }else{
            $result['success'] = 'error';
            $result['message'] = __('Invalid request.',WLPR_TEXT_DOMAIN);
        }
        wp_send_json($result);
    }

    /**
     * Process Migration
     * @param $start - limit start
     * @param $limit - limit
     * @return void
    */
    function doMigration($start,$limit){
        global $wpdb;
        if (!Woocommerce::hasAdminPrivilege()) {
            return;
        }
        $limit = (int) $limit;
        $start = (int) $start;
        $point_migration_action = new PointAction();
        $table = $point_migration_action->getTableName();
        $where = $wpdb->prepare('points > 0 ORDER BY id ASC LIMIT %d OFFSET %d;',array($limit,$start));
        $query = "SELECT * FROM {$table} WHERE {$where}";
        $migration_record = $wpdb->get_results($query, OBJECT);
        if(count($migration_record) > 0){
            $current_date = date('Y-m-d H:i:s');
            $point_setting = get_option('wlpr_settings');
            $expired_number = (isset($point_setting['wlpr_expired_after']) && !empty($point_setting['wlpr_expired_after'])) ? $point_setting['wlpr_expired_after'] : 0;
            $expired_period = (isset($point_setting['wlpr_expired_period']) && !empty($point_setting['wlpr_expired_period'])) ? $point_setting['wlpr_expired_period'] : 'DAY';
            $expired_date_after = '';
            if (!empty($expired_number)) {
                $expired_date_after = "+ $expired_number $expired_period";
            }

            $expired_email_date = (isset($point_setting['wlpr_expired_email_after']) && !empty($point_setting['wlpr_expired_email_after'])) ? $point_setting['wlpr_expired_email_after'] : 0;
            $expired_email_period = (isset($point_setting['wlpr_expired_email_period']) && !empty($point_setting['wlpr_expired_email_period'])) ? $point_setting['wlpr_expired_email_period'] : 'DAY';
            $expired_email_date_after = '';
            if (!empty($expired_email_date)) {
                $expired_email_date_after = "+ $expired_email_date $expired_email_period";
            }

            foreach ($migration_record as $record){
                $expire_status = 'in-active';
                $expire_date = NULL;
                $remainig_point = $record->points;
                if(!empty($expired_date_after)){
                    $expire_date = date('Y-m-d H:i:s', strtotime($record->created_date.' '.$expired_date_after));
                    if($expire_date < $current_date){
                        $expire_status = 'expired';
                        $remainig_point = 0;
                    }else{
                        $expire_status = 'active';
                    }
                }
                $email_date = null;
                if(!empty($expired_email_date_after)){
                    $email_date = date('Y-m-d H:i:s', strtotime($record->created_date.' '.$expired_email_date_after));
                }
                $point_migration_action->updateRow(array(
                    'expire_status' => $expire_status,
                    'expire_mail_date' => $email_date,
                    'expire_date' => $expire_date,
                    'remaining_point' => $remainig_point
                ), array('id' => $record->id));
            }
        }
    }

    function new_customer_popup(){
        $wlpr_none = (string)self::$input->post_get('wlpr_none','');
        if (!Woocommerce::hasAdminPrivilege() || !Woocommerce::verify_nonce($wlpr_none,'wlpr-popup-none')) {
            $result = array(
                'success' => 'failed'
            );
            wp_send_json($result);
        }else{
            $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'action' => 'wlpr_save_customer')));
            $page_details = array(
                'base_url' => $base_url,
                'wlpr_none' => Woocommerce::create_nonce('wlpr-new-customer')
            );
            $html = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/new_customer.php', $page_details)->render();
            $result = array(
                'success' => 'completed',
                'html' => $html
            );
            wp_send_json($result);
        }
    }

    function wlpr_save_customer(){
        global $wpdb;
        $wlpr_none = (string)self::$input->post_get('wlpr_none', '');
        $customer_validation = Validation::validateNewCustomer($_REQUEST);
        if (Woocommerce::hasAdminPrivilege() && Woocommerce::verify_nonce($wlpr_none,'wlpr-new-customer') && !is_array($customer_validation)) {
            $user_email = self::$input->post('user_email', '');
            $point = (int)self::$input->post('points', 0);
            $ref_code = self::$input->post('refer_code','');
            $user_email = sanitize_email($user_email);
            if($point > 0){
                $point_model = new Points();
                $where = $wpdb->prepare('user_email = %s', array($user_email));
                $table_point = $point_model->getWhere($where, 'points', true);
                if (empty($table_point)) {
                    $referralHelper = Loyalty::referral();
                    $uniqueReferCode = $referralHelper->get_unique_refer_code($ref_code);
                    $_data = array(
                        'user_email' => $user_email,
                        'points' => $point,
                        'created_date' => date("Y-m-d h:i:s"),
                        'refer_code' => $uniqueReferCode
                    );
                    if($point_model->insertRow($_data)){
                        // required log parameters
                        $redeem_price = 0.0;
                        $point_helper = Loyalty::point();
                        $earn_price = $point_helper->getEarnPointAmount($point);
                        $args = array(
                            'user_email' => $user_email,
                            'points' => $point,
                            'event_type' => 'admin-adjustment',
                            'created_date' => $_data['created_date'],
                            'redeem_price' => $redeem_price,
                            'earn_price' => $earn_price
                        );
                        if (is_admin()) {
                            $admin_user = wp_get_current_user();
                            $args['admin_user_id'] = $admin_user->ID;
                        }
                        // log the event
                        $point_log = new PointLog();
                        $point_log->add_log_entry($args);
                        $point_action = new PointAction();
                        $point_action->earn_point_action($user_email, 'admin-adjustment', $args);
                        $response = array(
                            'success' => true,
                            'redirect' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer'))),
                            'message' => esc_html__('Customer added successfully', WLPR_TEXT_DOMAIN)
                        );
                    }else{
                        $response = array(
                            'error' => true,
                            'message' => esc_html__('Customer save failed', WLPR_TEXT_DOMAIN)
                        );
                    }
                }else{
                    $response = array(
                        'error' => true,
                        'message' => esc_html__("Customer already exits", WLPR_TEXT_DOMAIN)
                    );
                }
            }else{
                $response = array(
                    'error' => true,
                    'message' => esc_html__("Point value must be greater then zero", WLPR_TEXT_DOMAIN)
                );
            }
        }elseif(is_array($customer_validation)){
            $message = '';
            foreach ($customer_validation as $field => $messages){
                $message .= str_replace('Wlpr','',implode(',',$messages));
            }
            $response = array(
                'error' => true,
                'message' => $message
            );
        }else{
            $response = array(
                'error' => true,
                'message' => esc_html__("You don't have access to save customer", WLPR_TEXT_DOMAIN)
            );
        }
        wp_send_json($response);
    }

    /**
     * Page task
     * @since 1.0.0
     */
    function managePages()
    {
        global $wpdb;
        if (!Woocommerce::hasAdminPrivilege()) {
            wp_die(__("Don't have access permission", WLPR_TEXT_DOMAIN));
        }
        if (self::$input->get('page', NULL) == WLPR_PLUGIN_SLUG) {
            $view = (string)self::$input->get('view', 'dashboard');
            $view = Validation::validateInputAlpha($view);
            $main_page_params = array(
                'current_view' => $view,
                'tab_content' => NULL,
                'extra' => $this->licenseValidationForm()
            );
            $is_already_migration = get_option('wlpr_already_migration',0);
            $migrate_html = '';
            if($is_already_migration == 0){
                if(!in_array($view,array('migration'))){
                    $view = 'dashboard';
                    $main_page_params['current_view'] = $view;
                }
                // count number of data
                $point_migration_action = new PointAction();
                $expire_migration_action = $point_migration_action->getWhere('points > 0', '*', false);
                if(count($expire_migration_action) > 0){
                    $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'migration')));
                    $migrate_html = __('Allow User Migration from Old version ',WLPR_TEXT_DOMAIN);
                    $migrate_html .= '<a href="'.$base_url.'" class="button lpr-update">'.__('Migrate',WLPR_TEXT_DOMAIN).'</a>';
                }else{
                    update_option('wlpr_already_migration',1);
                }
            }
            //update_option('wlpr_already_migration',0);
            switch ($view) {
                case 'migration':
                    //case 1: old user...
                    //  check migration already done... else do migration
                    //case 2: New install
                    // No need migration
                    $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
                    if($is_already_migration){
                        wp_redirect($base_url);
                    }else{
                        $point_migration_action = new PointAction();
                        $total_count = $point_migration_action->getWhere('points > 0', 'count(*) as total_count', true);
                        $page_details = array(
                            'base_url' => $base_url,
                            'total_count' => $total_count->total_count,
                            'process_count' => 0,
                            'limit' => 20,
                            'wlpr_none' => Woocommerce::create_nonce('wlpr-process-migration')
                        );
                        // count number of records
                        self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/process_migration.php', $page_details)->display();
                    }
                    break;

                case 'referral':
                    $referral_model = new Referral();
                    $search = (string)self::$input->post_get('search', '');
                    $search = sanitize_text_field($search);
                    $filter_order = (string)self::$input->post_get('filter_order', 'id');
                    $filter_order_dir = (string)self::$input->post_get('filter_order_dir', 'DESC');
                    $per_page = (int)self::$input->get('per_page', 5);
                    $current_page = (int)self::$input->get('page_number', 1);
                    $error_array = Validation::validateCommonFields($_REQUEST);
                    if(is_array($error_array)){
                        $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
                        wp_redirect($base_url);
                        exit;
                    }
                    $offset = $per_page * ($current_page - 1);

                    $ref_where = 'id > 0';
                    if (!empty($search)) {
                        $search_key = '%' . $search . '%';
                        $ref_where .= $wpdb->prepare(' AND (referred_by like %s OR new_user like %s)', array($search_key, $search_key));
                    }
                    $referral_order_by_sql    = sanitize_sql_orderby( "{$filter_order} {$filter_order_dir}" );
                    $referral_order_by = '';
                    if(!empty($referral_order_by_sql)){
                        $referral_order_by = " ORDER BY {$referral_order_by_sql}";
                    }
                    $where = $ref_where . $referral_order_by.$wpdb->prepare(' LIMIT %d OFFSET %d', array($per_page, $offset));
                    $referral_users = $referral_model->getWhere($where, '*', false);
                    $total_count = $referral_model->getWhere($ref_where, 'COUNT( DISTINCT id) as total_count', true);
                    $params = array(
                        'totalRows' => $total_count->total_count,
                        'perPage' => $per_page,
                        'baseURL' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'referral'))),
                        'currentPage' => (int)self::$input->get('page_number', 0),
                    );
                    $pagination = new Pagination($params);

                    $page_details = array(
                        'options' => get_option('wlpr_referral'),
                        'save_key' => 'wlpr_reference',
                        'referral_users' => $referral_users,
                        'base_url' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'referral'))),
                        'search' => $search,
                        'filter_order' => $filter_order,
                        'filter_order_dir' => $filter_order_dir,
                        'pagination' => $pagination,
                        'per_page' => $per_page,
                        'page_number' => $current_page,
                        'wlpr_referral_none' => Woocommerce::create_nonce('wlpr-referral-none')
                    );
                    $main_page_params['tab_content'] = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/referral.php', $page_details)->render();
                    self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/main.php', $main_page_params)->display();
                    break;
                case 'settings':
                    $current_status = get_option("wlpr_point_license_key_status", "invalid");
                    $license_key_error = true;
                    if ($current_status == "invalid") {
                        $license_key_message = __('Please enter valid license key!', WLPR_TEXT_DOMAIN);
                    } elseif ($current_status == "expired") {
                        $license_key_message = __('Your license key was expired!', WLPR_TEXT_DOMAIN);
                    } else {
                        $license_key_error = false;
                        $license_key_message = __('Your license key validated successfully!', WLPR_TEXT_DOMAIN);
                    }
                    $all_roles = wp_roles()->roles;
                    $option_rules = array();
                    foreach ($all_roles as $role_key => $role) {
                        $option_rules[$role_key] = $role['name'];
                    }

                    //launcher url

                    $launcher_url = \Wlpr_Customizer::get_customizer_url();
                    $page_details = array(
                        'options' => get_option('wlpr_settings'),
                        'license_key' => get_option('wlpr_point_license_key'),
                        'save_key' => 'wlpr_settings',
                        'license_key_message' => $license_key_message,
                        'license_key_error' => $license_key_error,
                        'exclude_earn_point_roles' => $option_rules,
                        'launcher_url' => $launcher_url,
                        'wlpr_setting_none' => Woocommerce::create_nonce('wlpr-setting-none')
                    );
                    $main_page_params['tab_content'] = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/settings.php', $page_details)->render();
                    self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/main.php', $main_page_params)->display();
                    break;
                case 'point_logs':
                    $point_log = new PointLog();
                    $per_page = (int)self::$input->get('per_page', 5);
                    $current_page = (int)self::$input->get('page_number', 1);
                    $search = (string)self::$input->post_get('search', '');
                    $filter_order = (string)self::$input->post_get('filter_order', 'id');
                    $filter_order_dir = (string)self::$input->post_get('filter_order_dir', 'DESC');
                    $validate_data =  Validation::validateCommonFields($_REQUEST);
                    if(is_array($validate_data)){
                        $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer')));
                        wp_redirect($base_url);
                        exit;
                    }
                    $offset = $per_page * ($current_page - 1);
                    $log_where = $wpdb->prepare('id > %d AND type != %s AND type != %s', array(0, 'private_note','expire_email'));
                    if (!empty($search)) {
                        $search = sanitize_text_field($search);
                        $search_key = '%' . $search . '%';
                        $log_where .= $wpdb->prepare(' AND (user_email like %s OR points like %s)', array($search_key, $search_key));
                    }
                    $point_log_order_by_sql    = sanitize_sql_orderby( "{$filter_order} {$filter_order_dir}" );
                    $log_order_by = '';
                    if(!empty($point_log_order_by_sql)){
                        $log_order_by = " ORDER BY {$point_log_order_by_sql}";
                    }
                    $where = $log_where .$log_order_by.$wpdb->prepare( ' LIMIT %d OFFSET %d', array($per_page, $offset));
                    $items = $point_log->getWhere($where, '*', false);
                    //$total_where = $wpdb->prepare('id > %d AND type != %s', array(0, 'private_note'));
                    $total_count = $point_log->getWhere($log_where, 'COUNT( DISTINCT id) as total_count', true);
                    $params = array(
                        'totalRows' => $total_count->total_count,
                        'perPage' => $per_page,
                        'baseURL' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'point_logs'))),
                        'currentPage' => self::$input->get('page_number', 0),
                    );
                    $pagination = new Pagination($params);
                    $page_details = array(
                        'items' => $items,
                        'point_helper' => Loyalty::point(),
                        'pagination' => $pagination,
                        'base_url' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'point_logs'))),
                        'per_page' => $per_page,
                        'page_number' => $current_page,
                        'search' => $search,
                        'filter_order' => $filter_order,
                        'filter_order_dir' => $filter_order_dir
                    );
                    self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/point_logs.php', $page_details)->display();
                    break;
                case 'customer':
                    $point_user = new Points();
                    $per_page = (int)self::$input->get('per_page', 5);
                    $current_page = (int)self::$input->get('page_number', 1);
                    $search = (string)self::$input->post_get('search', '');
                    $filter_order = (string)self::$input->get('filter_order', 'id');
                    $filter_order_dir = (string)self::$input->get('filter_order_dir', 'DESC');

                    $offset = $per_page * ($current_page - 1);
                    $validate_data =  Validation::validateCommonFields($_REQUEST);
                    if(is_array($validate_data)){
                        $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer')));
                        wp_redirect($base_url);
                    }
                    $customer_where = 'id > %d';
                    $customer_where_arg = array(0);
                    if (!empty($search)) {
                        $search = sanitize_text_field($search);
                        $customer_where .= $wpdb->prepare(' AND (user_email like %s OR refer_code like %s)',array('%' . $search . '%','%' . $search . '%'));
                    }

                    $customer_order_by_sql    = sanitize_sql_orderby( "{$filter_order} {$filter_order_dir}" );
                    if(!empty($customer_order_by_sql)){
                        $customer_where .= " ORDER BY {$customer_order_by_sql}";
                    }
                    $where = $wpdb->prepare($customer_where, $customer_where_arg);
                    if (isset($per_page) && isset($offset)) {
                        $where = $where . ' ' . $wpdb->prepare('LIMIT %d OFFSET %d', array($per_page, $offset));
                    }
                    $items = $point_user->getWhere($where, '*', false);
                    $total_where = $wpdb->prepare($customer_where, $customer_where_arg);
                    $total_count = $point_user->getWhere($total_where, 'COUNT( DISTINCT id) as total_count', true);
                    $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer')));
                    $params = array(
                        'totalRows' => $total_count->total_count,
                        'perPage' => $per_page,
                        'baseURL' => $base_url,
                        'currentPage' => $current_page,
                    );
                    $pagination = new Pagination($params);
                    $page_details = array(
                        'items' => $items,
                        'point_helper' => Loyalty::point(),
                        'pagination' => $pagination,
                        'base_url' => $base_url,
                        'per_page' => $per_page,
                        'page_number' => $current_page,
                        'search' => $search,
                        'filter_order' => $filter_order,
                        'filter_order_dir' => $filter_order_dir,
                        'files' => $this->download_files()
                    );
                    $main_page_params['tab_content'] .= self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/customer_point.php', $page_details)->render();
                    self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/main.php', $main_page_params)->display();
                    break;
                case 'customer_history':
                    $user_email = self::$input->get('user_email', '');
                    if (!empty($user_email) && filter_var($user_email, \FILTER_VALIDATE_EMAIL) !== false) {
                        $user_email = sanitize_email($user_email);
                        $t_filter_order = (string)self::$input->post_get('t_filter_order', 'id');
                        $t_filter_order_dir = (string)self::$input->post_get('t_filter_order_dir', 'DESC');
                        $t_search = (string)self::$input->post_get('t_search', '');
                        $e_filter_order = (string)self::$input->post_get('e_filter_order', 'created_date');
                        $e_filter_order_dir = (string)self::$input->post_get('e_filter_order_dir', 'ASC');
                        $e_search = (string)self::$input->post_get('e_search', '');
                        $per_page = (int)self::$input->get('per_page', 5);
                        $transaction_page_number = (int)self::$input->get('transaction_page_number', 1);
                        $action_page_number = (int)self::$input->get('action_page_number', 1);
                        $activity_page_number = (int)self::$input->get('activity_page_number', 1);

                        $validate_data =  Validation::validateCustomerHistory($_REQUEST);
                        if(is_array($validate_data)){
                            $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email)));
                            wp_redirect($base_url);
                            exit;
                        }
                        // filter
                        $filter_base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email)));
                        // user point details
                        $point_user = new Points();
                        $where = $wpdb->prepare('user_email = %s', array($user_email));
                        $user = $point_user->getWhere($where, '*', true);
                        // user transaction
                        $point_log = new PointLog();
                        $transaction_offset = $per_page * ($transaction_page_number - 1);
                        $transaction_where = $wpdb->prepare('user_email = %s AND type != %s AND type != %s', array($user_email, 'private_note','expire_email'));
                        if (!empty($t_search)) {
                            $t_search = sanitize_text_field($t_search);
                            $t_search_like = '%' . $t_search . '%';
                            $transaction_where .= $wpdb->prepare(' AND (user_email LIKE %s OR type LIKE %s OR points LIKE %s)', array($t_search_like, $t_search_like, $t_search_like));
                        }

                        $transaction_order_by_sql    = sanitize_sql_orderby( "{$t_filter_order} {$t_filter_order_dir}" );
                        $transaction_order_by = '';
                        if(!empty($transaction_order_by_sql)){
                            $transaction_order_by = " ORDER BY {$transaction_order_by_sql}";
                        }
                        $transaction_where_query = $transaction_where . $transaction_order_by. $wpdb->prepare( ' LIMIT %d OFFSET %d', array($per_page, $transaction_offset));
                        //$transaction_where = $wpdb->prepare('user_email = %s AND type != %s ORDER BY id DESC LIMIT %d OFFSET %d', array($user_email, 'private_note', $per_page, $transaction_offset));
                        $transaction = $point_log->getWhere($transaction_where_query, '*', false);
                        $transaction_total_where = $transaction_where;
                        $transaction_total_count = $point_log->getWhere($transaction_total_where, 'COUNT( DISTINCT id) as total_count', true);
                        $log_params = array(
                            'totalRows' => $transaction_total_count->total_count,
                            'perPage' => $per_page,
                            'baseURL' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email, 'action_page_number' => $action_page_number, 'activity_page_number' => $activity_page_number))),
                            'currentPage' => $transaction_page_number,
                            'queryStringSegment' => 'transaction_page_number'
                        );
                        $transaction_pagination = new Pagination($log_params);
                        // on going expired point
                        $point_action = new PointAction();
                        $action_offset = $per_page * ($action_page_number - 1);
                        $current_date = date("Y-m-d h:i:s");
                        $point_action_where = $wpdb->prepare('user_email = %s AND points > 0 AND expire_date >= %s AND expire_status = %s', array($user_email,$current_date,'active'));
                        if (!empty($e_search)) {
                            $e_search = sanitize_text_field($e_search);
                            $e_search_like = '%' . $e_search . '%';
                            $point_action_where .= $wpdb->prepare(' AND (user_email LIKE %s OR action LIKE %s OR points LIKE %s)', array($e_search_like, $e_search_like, $e_search_like));
                        }
                        $e_order_by_sql    = sanitize_sql_orderby( "{$e_filter_order} {$e_filter_order_dir}" );
                        $e_order_by = '';
                        if(!empty($e_order_by_sql)){
                            $e_order_by = " ORDER BY {$e_order_by_sql}";
                        }
                        $action_where_query = $point_action_where . $e_order_by. $wpdb->prepare(' LIMIT %d OFFSET %d', array($per_page, $action_offset));
                        //$point_action_where = $wpdb->prepare('user_email = %s AND points > 0 ORDER BY created_date ASC LIMIT %d OFFSET %d', array($user_email, $per_page, $action_offset));
                        $point_action_data = $point_action->getWhere($action_where_query, '*', false);
                        $action_total_where = $point_action_where;//$wpdb->prepare('user_email = %s AND points > 0 ORDER BY id DESC', array($user_email));
                        $action_total_count = $point_action->getWhere($action_total_where, 'COUNT( DISTINCT id) as total_count', true);
                        $action_params = array(
                            'totalRows' => $action_total_count->total_count,
                            'perPage' => $per_page,
                            'baseURL' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email, 'transaction_page_number' => $transaction_page_number, 'activity_page_number' => $activity_page_number))),
                            'currentPage' => $action_page_number,
                            'queryStringSegment' => 'action_page_number'
                        );
                        $action_pagination = new Pagination($action_params);

                        // activity
                        $per_page = (int)self::$input->get('per_page', 5);
                        $activity_offset = $per_page * ($activity_page_number - 1);
                        $activity_where = $wpdb->prepare('user_email = %s ORDER BY id DESC LIMIT %d OFFSET %d', array($user_email, $per_page, $activity_offset));
                        $activity = $point_log->getWhere($activity_where, '*', false);
                        $activity_total_where = $wpdb->prepare('user_email = %s ORDER BY id DESC', array($user_email));
                        $activity_total_count = $point_log->getWhere($activity_total_where, 'COUNT( DISTINCT id) as total_count', true);
                        $activity_params = array(
                            'totalRows' => $activity_total_count->total_count,
                            'perPage' => $per_page,
                            'baseURL' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email, 'action_page_number' => $action_page_number, 'transaction_page_number' => $transaction_page_number))),
                            'currentPage' => $activity_page_number,
                            'queryStringSegment' => 'activity_page_number'
                        );
                        $activity_pagination = new Pagination($activity_params);
                        $page_details = array(
                            'user_details' => $user,
                            'transaction' => $transaction,
                            'transaction_pagination' => $transaction_pagination,
                            'transaction_page_number' => $transaction_page_number,
                            'action_pagination' => $action_pagination,
                            'action_page_number' => $action_page_number,
                            'point_action' => $point_action_data,
                            'activity' => $activity,
                            'activity_pagination' => $activity_pagination,
                            'activity_page_number' => $activity_page_number,
                            'point_helper' => Loyalty::point(),
                            //'expired_date_after' => $expired_date_after,
                            'user_email' => $user_email,
                            'filter_base_url' => $filter_base_url,
                            'per_page' => $per_page,
                            't_filter_order' => $t_filter_order,
                            't_filter_order_dir' => $t_filter_order_dir,
                            't_search' => $t_search,
                            'e_filter_order' => $e_filter_order,
                            'e_filter_order_dir' => $e_filter_order_dir,
                            'e_search' => $e_search
                        );
                        self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/customer_history.php', $page_details)->display();
                        break;
                    }
                default:
                case 'dashboard':
                    $this->createRequiredTable();
                    $filter_date = (string)self::$input->post('filter_date', '90_days');
                    $null_date = '0000:00:00 00:00:00';
                    if ($filter_date == 'custom') {
                        $from_date = self::$input->post('from_date', $null_date);
                        $to_date = self::$input->post('to_date', $null_date);
                    }

                    $validate_data =  Validation::validateDashboard($_POST);
                    if(is_array($validate_data)){
                        $base_url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'dashboard')));
                        wp_redirect($base_url);
                        exit;
                    }
                    $point_log = new PointLog();

                    $timezone = new DateTimeZone('UTC');
                    $point_x = esc_html__('Month', WLPR_TEXT_DOMAIN);
                    $start = '';
                    $end = '';
                    $point_chart_data = array();
                    $offset = self::$woocommerce->current_offset();
                    try {
                        if ($filter_date == '90_days') {
                            $current_time = new DateTime('now', $timezone);
                            $last_time = new DateTime('-90 days', $timezone);
                            $end = $current_time->format('Y-m-d H:i:s');
                            $start = $last_time->format('Y-m-d H:i:s');
                            $point_x = esc_html__('Month', WLPR_TEXT_DOMAIN);
                            $point_chart_data[] = array(esc_html__('Month', WLPR_TEXT_DOMAIN), esc_html__('Points Awarded', WLPR_TEXT_DOMAIN),
                                esc_html__('Point Redeemed', WLPR_TEXT_DOMAIN));
                            $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%m-%Y\')';
                        } elseif ($filter_date == 'this_month') {
                            $current_time = new DateTime('now', $timezone);
                            $start = $current_time->format('Y-m-01 00:00:00');
                            $end = $current_time->format('Y-m-d H:i:s');
                            $point_x = esc_html(__('Date', WLPR_TEXT_DOMAIN));
                            $point_chart_data[] = array(esc_html(__('Date', WLPR_TEXT_DOMAIN)), esc_html(__('Points Awarded', WLPR_TEXT_DOMAIN)),
                                esc_html(__('Point Redeemed', WLPR_TEXT_DOMAIN)));
                            $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%d-%m-%Y\')';
                        } elseif ($filter_date == 'last_month') {
                            $current_time = new DateTime('-1 month', $timezone);
                            $start = $current_time->format('Y-m-01 00:00:00');
                            $end = $current_time->format('Y-m-t 23:59:59');
                            $point_x = esc_html(__('Date', WLPR_TEXT_DOMAIN));
                            $point_chart_data[] = array(esc_html(__('Date', WLPR_TEXT_DOMAIN)), esc_html(__('Points Awarded', WLPR_TEXT_DOMAIN)),
                                esc_html(__('Point Redeemed', WLPR_TEXT_DOMAIN)));
                            $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%d-%m-%Y\')';
                        } elseif ($filter_date == 'last_year') {
                            $current_time = new DateTime('-1 year', $timezone);
                            $start = $current_time->format('Y-01-01 00:00:00');
                            $end = $current_time->format('Y-12-t 23:59:59');
                            $point_x = esc_html(__('Month', WLPR_TEXT_DOMAIN));
                            $point_chart_data[] = array(esc_html(__('Month', WLPR_TEXT_DOMAIN)), esc_html(__('Points Awarded', WLPR_TEXT_DOMAIN)),
                                esc_html(__('Point Redeemed', WLPR_TEXT_DOMAIN)));
                            $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%m-%Y\')';
                        } elseif ($filter_date == 'custom') {
                            if ($to_date != $null_date) {
                                $end = self::$woocommerce->convert_wp_time_to_utc($to_date, 'Y-m-d H:i:00');
                            } else {
                                $end = $null_date;
                            }
                            if ($from_date != $null_date) {
                                $start = self::$woocommerce->convert_wp_time_to_utc($from_date, 'Y-m-d H:i:00');
                            } else {
                                $start = $null_date;
                            }
                            $point_x = esc_html(__('Date', WLPR_TEXT_DOMAIN));
                            $point_chart_data[] = array(esc_html(__('Date', WLPR_TEXT_DOMAIN)), esc_html(__('Points Awarded', WLPR_TEXT_DOMAIN)),
                                esc_html(__('Point Redeemed', WLPR_TEXT_DOMAIN)));
                            $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%d-%m-%Y\')';
                        }
                    } catch (Exception $e) {
                        // do nothing
                    }
                    if (empty($group_by)) {
                        $point_x = esc_html(__('Month', WLPR_TEXT_DOMAIN));
                        $point_chart_data[] = array(esc_html(__('Month', WLPR_TEXT_DOMAIN)), esc_html(__('Points Awarded', WLPR_TEXT_DOMAIN)),
                            esc_html(__('Point Redeemed', WLPR_TEXT_DOMAIN)));
                        $group_by = 'DATE_FORMAT(CONVERT_TZ(created_date,\'+00:00\',\'' . $offset . '\'),\'%d-%m-%Y\')';
                    }
                    $analsis_where = '';
                    $referral_where = '';
                    if (!empty($start) || !empty($end)) {
                        $analsis_where = $wpdb->prepare('(created_date >= %s OR created_date = %s) AND (created_date <= %s OR created_date = %s) AND type != %s AND type != %s', array($start, $null_date, $end, $null_date, 'private_note','expire_email'));
                        $referral_where = $wpdb->prepare('(created_date >= %s OR created_date = %s) AND (created_date <= %s OR created_date = %s) AND (status = %s OR status = %s)', array($start, $null_date, $end, $null_date,'completed',''));
                    }
                    $analsis_data = $point_log->getWhere($analsis_where, 'SUM(CASE WHEN type 
                            IN("admin-adjustment","rest-api","order-placed","import","product-review","account-signup","referring_user","referral_point","birth_date","on_birth_date","facebook_share","twitter_share","email_share","whatsapp_share") AND points > 0 THEN points ELSE 0 END) as total_positive_earn_point_rewards,
                            SUM(CASE WHEN type 
                            IN("admin-adjustment","rest-api","import","order-cancelled","order-refunded") AND points < 0 THEN points ELSE 0 END) as total_negative_earn_point_rewards,
                            SUM(CASE WHEN type IN("order-redeem") AND points < 0 THEN points ELSE 0 END) as total_point_redeemed, 
                            SUM(CASE WHEN type IN("order-cancelled","order-refunded") AND points > 0 THEN points ELSE 0 END) as total_cancel_point_redeemed, 
                            COUNT(DISTINCT user_email) as total_customer,
                            SUM(CASE WHEN type IN("order-redeem") AND redeem_price > 0 THEN redeem_price ELSE 0 END) as total_redeem_price,
                            SUM(CASE WHEN type IN("order-cancelled","order-refunded") AND points > 0 THEN redeem_price ELSE 0 END) as total_cancel_redeem_price,
                             SUM(CASE WHEN earn_price > 0 THEN earn_price ELSE 0 END) as total_earn_price', true);
                    $referral_model = new Referral();
                    $referral_data = $referral_model->getWhere($referral_where, 'SUM(referred_order_price) as total_ref_price', true);
                    // first check year, then month, then date
                    $cart_query = 'SELECT (' . $group_by . ') as chart_x, SUM(CASE WHEN type 
                            IN("admin-adjustment","rest-api","order-placed","import","product-review","account-signup","referring_user","referral_point","birth_date","on_birth_date","facebook_share","twitter_share","email_share","whatsapp_share") AND points > 0 THEN points ELSE 0 END) as total_positive_earn_point_rewards,
                            SUM(CASE WHEN type 
                            IN("admin-adjustment","rest-api","import","order-cancelled","order-refunded") AND points < 0 THEN points ELSE 0 END) as total_negative_earn_point_rewards,
                            SUM(CASE WHEN type IN("order-redeem") AND points < 0 THEN points ELSE 0 END) as total_point_redeemed, 
                            SUM(CASE WHEN type IN("order-cancelled","order-refunded") AND points > 0 THEN points ELSE 0 END) as total_cancel_point_redeemed
                             FROM ' . $point_log->getTableName() . ' WHERE ' . $analsis_where;
                    $cart_query .= ' GROUP BY ' . $group_by;
                    $chart_data = $point_log->rawQuery($cart_query, false);
                    if (!empty($chart_data)) {
                        foreach ($chart_data as $chart) {
                            $chart_earn_point = ((isset($chart->total_positive_earn_point_rewards) && $chart->total_positive_earn_point_rewards) ? $chart->total_positive_earn_point_rewards : 0);
                            $chart_earn_point += ((isset($chart->total_negative_earn_point_rewards)) ? (float)$chart->total_negative_earn_point_rewards : 0);
                            if($chart_earn_point < 0){
                                $chart_earn_point = 0;
                            }
                            $chart_redem_point = (isset($chart->total_point_redeemed)) ? $chart->total_point_redeemed : 0;
                            $chart_redem_point += (isset($chart->total_cancel_point_redeemed) && $chart->total_cancel_point_redeemed) ? $chart->total_cancel_point_redeemed : 0;
                            if($chart_redem_point < 0){
                                $chart_redem_point = -$chart_redem_point;
                            }else{
                                $chart_redem_point = 0;
                            }
                            $chart_point_data = array(
                                $chart->chart_x,
                                (float)$chart_earn_point,
                                (float)$chart_redem_point
                            );
                            if (!empty($chart_point_data)) {
                                $point_chart_data[] = $chart_point_data;
                            }
                        }
                    }
                    $date_filter = array(
                        '90_days' => esc_html(__('Last 90 days', WLPR_TEXT_DOMAIN)),
                        'this_month' => esc_html(__('This month', WLPR_TEXT_DOMAIN)),
                        'last_month' => esc_html(__('Last month', WLPR_TEXT_DOMAIN)),
                        'last_year' => esc_html(__('Last year', WLPR_TEXT_DOMAIN)),
                        'custom' => esc_html(__('Custom date', WLPR_TEXT_DOMAIN))
                    );
                    if (count($point_chart_data) <= 1) {
                        $point_chart_data = array();
                    }
                    $page_details = array(
                        'analsis_data' => $analsis_data,
                        'referral_data' => $referral_data,
                        'point_chart_data' => $point_chart_data,
                        'point_y' => esc_html(__('Points', WLPR_TEXT_DOMAIN)),
                        'point_x' => $point_x,
                        'filter_date' => $filter_date,
                        'start' => $start,
                        'end' => $end,
                        'date_filter' => $date_filter,
                        'point_helper' => self::$point,
                        'woocommerce_helper' => self::$woocommerce
                    );
                    $main_page_params['tab_content'] = self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/point_analysis.php', $page_details)->render();
                    $where = $wpdb->prepare('id > %d AND type != %s AND type != %s ORDER BY id DESC LIMIT 5', array(0, 'private_note', 'expire_email'));
                    $items = $point_log->getWhere($where, '*', false);
                    $page_details = array(
                        'items' => $items,
                        'point_helper' => Loyalty::point()
                    );
                    $main_page_params['tab_content'] .= self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/recent_point.php', $page_details)->render();
                    $main_page_params['migrate_content']    = $migrate_html;
                    self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/main.php', $main_page_params)->display();
                    break;
            }
        } else {
            wp_die(__('Unable to process', WLPR_TEXT_DOMAIN));
        }
    }

    /**
     * License validation form
     * @return string
     */
    function licenseValidationForm()
    {
        if (!Woocommerce::hasAdminPrivilege()) {
            wp_die(__("Don't have permission to access this page", WLPR_TEXT_DOMAIN));
        }
        $license_helper = Loyalty::licence();
        if ($license_helper->isValidLicense()) {
            return NULL;
        } else {
            $license_key = $license_helper->getLicenseKey();
            return '<div class="model-license-key">
            <form id="validate_license_key_form_modal">
                <div class="wecp-card">
                    <div class="pt-1">
                        <h2 class="swal2-title">' . __("Enter License Key", WLPR_TEXT_DOMAIN) . '</h2>   
                        <input type="hidden" name="action" value="wlpr_validate_license_key">
                        <input type="hidden" name="wlpr_none" value="'.Woocommerce::create_nonce('wlpr-license-none').'">
                        <input type="text" id="modal_license_key" name="license_key" class="license-key-input" value="' . $license_key . '"
                               placeholder="Enter license key here...">
                        <p class="license_key_message" style="text-align: left;margin: 10px 0 25px;"></p>
                        <button id="validate-_license_key_modal_btn" type="submit" class="button license-key-submit">' . __("Validate", WLPR_TEXT_DOMAIN) . '</button>
                    </div>
                </div>
            </form>
        </div>
        <div class="hvrbox-layer_top"></div>';
        }
    }

    /**
     * Validate the license key
     */
    function validateLicenseKey()
    {
        $license_key = (string)self::$input->post('license_key', NULL);
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        if (!Woocommerce::hasAdminPrivilege()) {
            $response['license_key'] = false;
            $response['license_key_message'] = esc_html__("Don't have permission to access this request.", WLPR_TEXT_DOMAIN);
        }else if(!Woocommerce::verify_nonce($wlpr_none,'wlpr-license-none')){
            $response['license_key'] = false;
            $response['license_key_message'] = esc_html__("Invalid request.", WLPR_TEXT_DOMAIN);
        }else{
            $response = Loyalty::licence()->verifyLicenseKey($license_key);
        }
        wp_send_json($response);
    }

    /**
     * save global setting
     * @since 1.0.0
     */
    function saveSettings()
    {
        $response = array();
        $save_not_allowed = true;
        if (!Woocommerce::hasAdminPrivilege()) {
            $save_not_allowed = false;
        }
        $key = (string)self::$input->post('option_key', NULL);
        $key = Validation::validateInputAlpha($key);
        $wlpr_none = (string)self::$input->post('wlpr_none', NULL);
        if ($save_not_allowed && !empty($key) && Woocommerce::verify_nonce($wlpr_none,'wlpr-setting-none')) {
            $data = self::$input->post();
            if (isset($data['option_key'])) {
                unset($data['option_key']);
            }
            if (isset($data['action'])) {
                unset($data['action']);
            }
            $license_key = self::$input->post('license_key', NULL);
            $response = Loyalty::licence()->verifyLicenseKey($license_key);
            $data = apply_filters('woocommerce_loyalty_points_rewards_before_save_settings', $data, $key);
            $validate_data = Validation::validateSettingsTab($_REQUEST);
            if(is_array($validate_data)){
                foreach ($validate_data as $field => $messages){
                    $validate_data[$field] = str_replace('Wlpr','',implode(',',$messages));
                }
                $response['error'] = true;
                $response['field_error'] = $validate_data;
                $response['message'] = __('Settings not saved!');
            }
            if(!isset($response['error']) || !$response['error']){
                foreach ($data as $d_key => $d_value){
                    if(in_array($d_key,array(
                        'wlpr_account_signup_message',
                        'wlpr_write_review_message',
                        'wlpr_single_product_message',
                        'wlpr_variable_product_message',
                        'wlpr_cart_earn_points_message',
                        'wlpr_cart_redeem_points_message',
                        'wlpr_thank_you_message',
                        'wlpr_earn_point_order_summary_text'
                    ))){
                        $d_value = stripslashes($d_value);
                        $data[$d_key] = Woocommerce::getCleanHtml($d_value);
                    }
                }
                if(isset($data['generate_api_key']) && $data['generate_api_key'] == 1){
                    $consumer_key    = 'ck_' . wc_rand_hash();
                    $consumer_secret = 'cs_' . wc_rand_hash();
                    $data['wlpr_client_id'] = $consumer_key;
                    $data['wlpr_client_secret'] = $consumer_secret;//wp-loyalty-points-rewards&view=settings
                    $response['redirect'] = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'settings')));
                }
                update_option($key, $data, true);
                do_action('woocommerce_loyalty_points_rewards_after_save_settings', $data, $key);

                $response['error'] = false;
                $response['message'] = esc_html__('Settings saved successfully!', WLPR_TEXT_DOMAIN);
            }
        } else {
            $response['error'] = true;
            $response['message'] = esc_html__('Settings not saved!', WLPR_TEXT_DOMAIN);
        }
        wp_send_json($response);
    }

    function applyPoint()
    {
        global $wpdb;
        $save_not_allowed = true;
        if (!Woocommerce::hasAdminPrivilege()) {
            $save_not_allowed = false;
        }
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        $user_email = self::$input->post('user_email', '');
        $point = self::$input->post('point', 0);
        if ($save_not_allowed && is_numeric($point) && !empty($user_email) && filter_var($user_email, \FILTER_VALIDATE_EMAIL) !== false
            && Woocommerce::verify_nonce($wlpr_none,'wlpr-apply-point-none')) {
            $point = (int)$point;
            $user_email = sanitize_email($user_email);
            if($point >= 0){
                $point_model = new Points();
                $where = $wpdb->prepare('user_email = %s', array($user_email));
                $table_point = $point_model->getWhere($where, 'points', true);
                $old_point = 0;
                if (isset($table_point->points) && !empty($table_point->points)) {
                    $old_point = $table_point->points;
                }
                $point_date = date("Y-m-d h:i:s");
                $point_model->updateRow(array(
                    'points' => $point,
                    'created_date' => $point_date,
                ), array('user_email' => $user_email));
                //update point log
                $point = $point - $old_point;
                $earn_price = 0;
                $redeem_price = 0;
                if($point > 0){
                    //find earn point from price
                    $earn_price = self::$point->getEarnPointAmount($point);
                }else{
                    $redeem_price = self::$point->getRedeemPointAmount(-$point);
                }
                $point_log = new PointLog();
                $point_log_data = array(
                    'user_email' => $user_email,
                    'points' => $point,
                    'event_type' => 'admin-adjustment',
                    'admin_user_id' => get_current_user_id(),
                    'created_date' => $point_date,
                    'redeem_price' => $redeem_price,
                    'earn_price' => $earn_price
                );
                $point_log->add_log_entry($point_log_data);
                $point_action = new PointAction();
                if($point > 0){
                    $point_action->earn_point_action($user_email, 'admin-adjustment', $point_log_data);
                }else{
                    $point_action->redeem_point_action($user_email, 'admin-adjustment', $point_log_data);
                }
                $response = array(
                    'success' => true,
                    'redirect' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history', 'user_email' => $user_email))),
                    'message' => esc_html__('Point updated successfully', WLPR_TEXT_DOMAIN)
                );
            }else{
                $response = array(
                    'error' => true,
                    'message' => esc_html__('Point update failed, Negative point not allowed', WLPR_TEXT_DOMAIN)
                );
            }

        } else {
            $response = array(
                'error' => true,
                'message' => esc_html__('Point update failed', WLPR_TEXT_DOMAIN)
            );
        }
        wp_send_json($response);
    }
    function bulkDeleteUser(){
        global $wpdb;
        $data = self::$input->post();
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        if(Woocommerce::verify_nonce($wlpr_none,'wlpr-delete-user-none') && Woocommerce::hasAdminPrivilege()){
            if(isset($data['cb']) && !empty($data['cb']) && is_array($data['cb'])){
                foreach ($data['cb'] as $cb){
                    $point_model = new Points();
                    $where = $wpdb->prepare('id = %d', array((int)$cb));
                    $user_table = $point_model->getWhere($where, 'user_email', true);
                    if(isset($user_table->user_email) && !empty($user_table->user_email)){
                        $condition = array(
                            'user_email' => sanitize_email($user_table->user_email)
                        );
                        $point_model->deleteRow($condition);
                        $point_log = new PointLog();
                        $point_log->deleteRow($condition);
                        $point_action = new PointAction();
                        $point_action->deleteRow($condition);
                    }
                }
                $response = array(
                    'success' => true,
                    'redirect' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer'))),
                    'message' => esc_html__('User deleted successfully', WLPR_TEXT_DOMAIN)
                );
            }else{
                $response = array(
                    'error' => true,
                    'message' => esc_html__("Please select at-least one checkbox", WLPR_TEXT_DOMAIN)
                );
            }
        }else{
            $response = array(
                'error' => true,
                'message' => esc_html__("you don't have delete permission", WLPR_TEXT_DOMAIN)
            );
        }

        wp_send_json($response);
    }
    /**
     * Delete user
     */
    function deleteUser(){
        $delete_not_allowed = true;
        if (!Woocommerce::hasAdminPrivilege()) {
            $delete_not_allowed = false;
        }
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        $user_email = self::$input->post('user_email', '');
        if ($delete_not_allowed && !empty($user_email) && filter_var($user_email, \FILTER_VALIDATE_EMAIL) !== false && Woocommerce::verify_nonce($wlpr_none,'wlpr-delete-user-none')) {
            $user_email = sanitize_email($user_email);
            $point_model = new Points();
            $condition = array(
                'user_email' => $user_email
            );
            $point_model->deleteRow($condition);
            $point_log = new PointLog();
            $point_log->deleteRow($condition);
            $point_action = new PointAction();
            $point_action->deleteRow($condition);
            $response = array(
                'success' => true,
                'redirect' => admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer'))),
                'message' => esc_html__('User deleted successfully', WLPR_TEXT_DOMAIN)
            );
        } else {
            $response = array(
                'error' => true,
                'message' => esc_html__('User delete failed', WLPR_TEXT_DOMAIN)
            );
        }
        wp_send_json($response);
    }

    /**
     * Render category point settings
     * @param $term
     * @since 1.0.0
     */
    function renderProductCategoryFields($term)
    {
        if (Woocommerce::hasAdminPrivilege()) {
            $term_id = (int)(isset($term->term_id) && !empty($term->term_id)) ? $term->term_id : 0;
            $wlpr_category_enable_earn_point = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_enable_earn_point', true);
            $earn_point_setting = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_points_earned', true);
            $category_points_max_discount = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_points_max_discount', true);
            $category_points_max_discount_type = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_points_max_discount_type', true);
            $max_product_discount_enabled = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_max_discount_enabled', true);
            $renewal_setting = self::$woocommerce->get_term_meta($term_id, '_wlpr_category_points_renewal_points', true);
            $page_details = array(
                'wlpr_category_enable_earn_point' => $wlpr_category_enable_earn_point,
                'earn_point_setting' => json_decode($earn_point_setting, true),
                'max_product_discount' => $category_points_max_discount,
                'max_product_discount_type' => $category_points_max_discount_type,
                'max_product_discount_enabled' => $max_product_discount_enabled,
                'renewal_setting' => json_decode($renewal_setting, true)
            );
            self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/category_settings.php', $page_details)->display();
        }
    }

    /**
     * Save category point setting
     * @param $term_id
     * @since 1.0.0
     */
    function saveProductCategoryPointsField($term_id)
    {
        if (Woocommerce::hasAdminPrivilege()) {
            $term_id = (int) $term_id;
            $post = self::$input->post();
            $validate_data = Validation::validateCategorySetting($_REQUEST);
            if(!is_array($validate_data)){
                // points earned
                if (isset($post['_wlpr_category_points_earned']) && '' !== $post['_wlpr_category_points_earned']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_points_earned', json_encode($post['_wlpr_category_points_earned']));
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_points_earned');
                }

                if (isset($post['_wlpr_category_enable_earn_point']) && '' !== $post['_wlpr_category_enable_earn_point']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_enable_earn_point', $post['_wlpr_category_enable_earn_point']);
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_enable_earn_point');
                }

                if (isset($post['_wlpr_category_points_max_discount']) && '' !== $post['_wlpr_category_points_max_discount']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_points_max_discount', $post['_wlpr_category_points_max_discount']);
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_points_max_discount');
                }
                if (isset($post['_wlpr_category_points_max_discount_type']) && '' !== $post['_wlpr_category_points_max_discount_type']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_points_max_discount_type', $post['_wlpr_category_points_max_discount_type']);
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_points_max_discount_type');
                }

                if (isset($post['_wlpr_category_max_discount_enabled']) && '' !== $post['_wlpr_category_max_discount_enabled']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_max_discount_enabled', $post['_wlpr_category_max_discount_enabled']);
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_max_discount_enabled');
                }

                if (isset($post['_wlpr_category_points_renewal_points']) && '' !== $post['_wlpr_category_points_renewal_points']) {
                    self::$woocommerce->update_term_meta($term_id, '_wlpr_category_points_renewal_points', json_encode($post['_wlpr_category_points_renewal_points']));
                } else {
                    self::$woocommerce->delete_term_meta($term_id, '_wlpr_category_points_renewal_points');
                }
            }
        }
    }

    /**
     * Render simple product settings
     * @since 1.0.0
     */
    function renderSimpleProductFields()
    {
        if (Woocommerce::hasAdminPrivilege()) {
            global $thepostid;
            $product_type = 'simple';
            $thepostid = (int) $thepostid;
            if(!empty($thepostid)){
                $product_type = \WC_Product_Factory::get_product_type($thepostid);
            }
            $wlpr_product_enable_earn_point = get_post_meta($thepostid, '_wlpr_product_enable_earn_point', true);
            $points_max_discount = get_post_meta($thepostid, '_wlpr_product_points_max_discount', true);
            $max_product_discount_enabled = get_post_meta($thepostid, '_wlpr_product_max_discount_enabled', true);
            $points_max_discount_type = get_post_meta($thepostid, '_wlpr_product_points_max_discount_type', true);
            $earn_point_setting = get_post_meta($thepostid, '_wlpr_product_points_earned', true);
            $renewal_earn_point_setting = get_post_meta($thepostid, '_wlpr_renewal_product_points_earned', true);
            $page_details = array(
                'wlpr_product_enable_earn_point' => $wlpr_product_enable_earn_point,
                'earn_point_setting' => json_decode($earn_point_setting, true),
                'max_product_discount_enabled' => $max_product_discount_enabled,
                'max_product_discount' => $points_max_discount,
                'max_product_discount_type' => $points_max_discount_type,
                'renewal_setting' => json_decode($renewal_earn_point_setting, true),
                'product_type' => $product_type
            );
            self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/product_settings.php', $page_details)->display();
        }

    }

    /**
     * Save simple product point settings
     * @param $post_id
     * @since 1.0.0
     */
    public function saveSimpleProductFields($post_id)
    {
        if (Woocommerce::hasAdminPrivilege()) {
            $post_id = (int)$post_id;
            $post = self::$input->post();
            $validate_data = Validation::validateSimpleProduct($_REQUEST);
            if(!is_array($validate_data)){
                if (isset($post['_wlpr_product_enable_earn_point']) && '' !== $post['_wlpr_product_enable_earn_point']) {
                    update_post_meta($post_id, '_wlpr_product_enable_earn_point', stripslashes($post['_wlpr_product_enable_earn_point']));
                } else {
                    delete_post_meta($post_id, '_wlpr_product_enable_earn_point');
                }

                if (isset($post['_wlpr_product_points_earned']) && '' !== $post['_wlpr_product_points_earned']) {
                    update_post_meta($post_id, '_wlpr_product_points_earned', json_encode($post['_wlpr_product_points_earned']));
                } else {
                    delete_post_meta($post_id, '_wlpr_product_points_earned');
                }

                if (isset($post['_wlpr_product_points_max_discount']) && '' !== $post['_wlpr_product_points_max_discount']) {
                    update_post_meta($post_id, '_wlpr_product_points_max_discount', stripslashes($post['_wlpr_product_points_max_discount']));
                } else {
                    delete_post_meta($post_id, '_wlpr_product_points_max_discount');
                }

                if (isset($post['_wlpr_product_max_discount_enabled']) && '' !== $post['_wlpr_product_max_discount_enabled']) {
                    update_post_meta($post_id, '_wlpr_product_max_discount_enabled', stripslashes($post['_wlpr_product_max_discount_enabled']));
                } else {
                    delete_post_meta($post_id, '_wlpr_product_max_discount_enabled');
                }

                if (isset($post['_wlpr_product_points_max_discount_type']) && '' !== $post['_wlpr_product_points_max_discount_type']) {
                    update_post_meta($post_id, '_wlpr_product_points_max_discount_type', stripslashes($post['_wlpr_product_points_max_discount_type']));
                } else {
                    delete_post_meta($post_id, '_wlpr_product_points_max_discount_type');
                }
                if (isset($post['_wlpr_renewal_product_points_earned']) && '' !== $post['_wlpr_renewal_product_points_earned']) {
                    update_post_meta($post_id, '_wlpr_renewal_product_points_earned', json_encode($post['_wlpr_renewal_product_points_earned']));
                } else {
                    delete_post_meta($post_id, '_wlpr_renewal_product_points_earned');
                }
            }
        }
    }

    /**
     * Render variation point settings
     * @param $loop
     * @param $variation_data
     * @param $variation
     * @since 1.0.0
     */
    public function renderVariableProductFields($loop, $variation_data, $variation)
    {
        if (Woocommerce::hasAdminPrivilege() && isset($variation->ID) && !empty($variation->ID)) {
            //_wlpr_product_points_earned
            $variation->ID = (int)$variation->ID;
            $earn_point_setting = json_decode(get_post_meta($variation->ID, '_wlpr_variable_product_points_earned', true), true);
            $product_type = 'variable';
            if(isset($variation->post_parent) && !empty($variation->post_parent)){
                $product_type = \WC_Product_Factory::get_product_type($variation->post_parent);
            }
            $wlpr_variable_product_enable_earn_point = get_post_meta($variation->ID, '_wlpr_variable_product_enable_earn_point', true);
            $max_discount = get_post_meta($variation->ID, '_wlpr_variable_product_points_max_discount', true);
            $max_variant_discount_enabled = get_post_meta($variation->ID, '_wlpr_variable_product_max_discount_enabled', true);
            $max_discount_type = get_post_meta($variation->ID, '_wlpr_variable_product_points_max_discount_type', true);
            $renewal_earn_point_setting = json_decode(get_post_meta($variation->ID, '_wlpr_variable_product_points_renewal_points', true), true);
            $page_details = array(
                'wlpr_variable_product_enable_earn_point' => $wlpr_variable_product_enable_earn_point,
                'earn_point_setting' => $earn_point_setting,
                'max_variant_discount_enabled' => $max_variant_discount_enabled,
                'max_discount' => $max_discount,
                'max_discount_type' => $max_discount_type,
                'renewal_earn_point_setting' => $renewal_earn_point_setting,
                'loop' => $loop,
                'variation_data' => $variation_data,
                'variation' => $variation,
                'product_type' => $product_type
            );
            self::$template->setData(WLPR_PLUGIN_PATH . 'App/Views/Admin/variable_product_fields.php', $page_details)->display();
            do_action('wlpr_point_after_variable_product_fields', $loop, $variation_data);
        }
    }

    /**
     * save variation point settings
     * @param $variation_id
     * @since 1.0.0
     */
    public function saveVariableProductFields($variation_id)
    {
        if (Woocommerce::hasAdminPrivilege()) {
            $post = self::$input->post();
            $variation_id = (int)$variation_id;
            // find the index for the given variation ID and save the associated points earned
            $index = array_search($variation_id, $post['variable_post_id']);
            if (false !== $index) {

                // points earned
                if (isset($post['_wlpr_variable_product_points_earned']) && '' !== $post['_wlpr_variable_product_points_earned'][$index]
                    && isset($post['_wlpr_variable_product_points_earned'][$index]['wlpr_earn_point']) && is_numeric($post['_wlpr_variable_product_points_earned'][$index]['wlpr_earn_point'])
                    && isset($post['_wlpr_variable_product_points_earned'][$index]['wlpr_point_earn_price']) && is_numeric($post['_wlpr_variable_product_points_earned'][$index]['wlpr_point_earn_price'])
                ) {
                    update_post_meta($variation_id, '_wlpr_variable_product_points_earned', json_encode($post['_wlpr_variable_product_points_earned'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_points_earned');
                }

                //enable earn point
                if (isset($post['_wlpr_variable_product_enable_earn_point']) && '' !== $post['_wlpr_variable_product_enable_earn_point'][$index]
                && in_array($post['_wlpr_variable_product_enable_earn_point'][$index], array('global', 'product','disable'))) {
                    update_post_meta($variation_id, '_wlpr_variable_product_enable_earn_point', stripslashes($post['_wlpr_variable_product_enable_earn_point'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_enable_earn_point');
                }

                //enable max discount
                if (isset($post['_wlpr_variable_product_max_discount_enabled']) && '' !== $post['_wlpr_variable_product_max_discount_enabled'][$index]
                    &&   in_array($post['_wlpr_variable_product_max_discount_enabled'][$index], array('yes', 'no'))
                ) {
                    update_post_meta($variation_id, '_wlpr_variable_product_max_discount_enabled', stripslashes($post['_wlpr_variable_product_max_discount_enabled'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_max_discount_enabled');
                }

                // maximum points discount
                if (isset($post['_wlpr_variable_product_points_max_discount']) && '' !== $post['_wlpr_variable_product_points_max_discount'][$index]
                    && is_numeric($post['_wlpr_variable_product_points_max_discount'][$index])
                ) {
                    update_post_meta($variation_id, '_wlpr_variable_product_points_max_discount', stripslashes($post['_wlpr_variable_product_points_max_discount'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_points_max_discount');
                }
                // maximum points discount type
                if (isset($post['_wlpr_variable_product_points_max_discount_type']) && '' !== $post['_wlpr_variable_product_points_max_discount_type'][$index]
                    && in_array($post['_wlpr_variable_product_points_max_discount_type'][$index], array('fixed', 'percentage'))
                ) {
                    update_post_meta($variation_id, '_wlpr_variable_product_points_max_discount_type', stripslashes($post['_wlpr_variable_product_points_max_discount_type'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_points_max_discount_type');
                }
                // change points for renewal
                if (isset($post['_wlpr_variable_product_points_renewal_points']) && '' !== $post['_wlpr_variable_product_points_renewal_points'][$index]
                    && isset($post['_wlpr_variable_product_points_renewal_points'][$index]['wlpr_earn_point']) && is_numeric($post['_wlpr_variable_product_points_renewal_points'][$index]['wlpr_earn_point'])
                    && isset($post['_wlpr_variable_product_points_renewal_points'][$index]['wlpr_point_earn_price']) && is_numeric($post['_wlpr_variable_product_points_renewal_points'][$index]['wlpr_point_earn_price'])
                ) {
                    update_post_meta($variation_id, '_wlpr_variable_product_points_renewal_points', json_encode($post['_wlpr_variable_product_points_renewal_points'][$index]));
                } else {
                    delete_post_meta($variation_id, '_wlpr_variable_product_points_renewal_points');
                }
            }
        }

    }


    /**
     * print the admin notices
     * @param $class
     * @param $message
     * @since 1.0.0
     */
    public function addAdminNotice($class, $message)
    {
        if (Woocommerce::hasAdminPrivilege()) {
            echo '<div class="' . esc_attr($class) . '" style="position:relative;"><p>';
            echo wp_kses($message, array('a' => array('href' => array(), 'target' => array())));
            echo '</p></div>';
        }
    }

    /**
     * save private customer note
     * @since 1.0.0
     */
    function savePrivateCustomerNote()
    {
        $private_note = (string)self::$input->post('private_note', '');
        $user_email = self::$input->post('user_email', '');
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        $response = array();
        //
        if (!Woocommerce::hasAdminPrivilege()) {
            $response['error'] = true;
            $response['message'] = esc_html__("you don't have access add private note.", WLPR_TEXT_DOMAIN);
        }else if(!Woocommerce::verify_nonce($wlpr_none,'wlpr-private-note-none')){
            $response['error'] = true;
            $response['message'] = esc_html__('Invalid request', WLPR_TEXT_DOMAIN);
        } else if(!filter_var($user_email, \FILTER_VALIDATE_EMAIL) !== false){
            $response['error'] = true;
            $response['message'] = esc_html__('Invalid request', WLPR_TEXT_DOMAIN);
        } else if (empty($private_note) || empty($user_email)) {
            $response['error'] = true;
            $response['message'] = esc_html__('You must enter private note field', WLPR_TEXT_DOMAIN);
        }else if (!empty($private_note) && !(bool)preg_match("/^[A-Za-z0-9_! \"#$%&'()*+{},\-.\\:\/;=?@^_]+$/", $private_note)) {
            $response['error'] = true;
            $response['message'] = esc_html__('Invalid private note.', WLPR_TEXT_DOMAIN);
        } else {
            try {
                // required log parameters
                $args = array(
                    'user_email' => sanitize_email($user_email),
                    'points' => 0,
                    'event_type' => 'private_note',
                    'created_date' => date('Y-m-d H:i:s'),
                    'log_data' => array('note' => Woocommerce::getCleanHtml($private_note)),
                    'redeem_price' => 0.0,
                    'earn_price' => 0.0
                );
                // log the event
                $point_log = new PointLog();
                $point_log->add_log_entry($args);
                $response['error'] = false;
                $response['message'] = esc_html__('Private note added successfully', WLPR_TEXT_DOMAIN);
                $url = admin_url('admin.php?' . http_build_query(array('page' => WLPR_PLUGIN_SLUG, 'view' => 'customer_history',
                        'user_email' => $user_email)));
                $response['redirect'] = $url;
            } catch (Exception $e) {
                $response['error'] = true;
                $response['message'] = esc_html__('Private note save failed', WLPR_TEXT_DOMAIN);
            }
        }
        wp_send_json($response);
    }

    /**
     * Save referral settings
     * @since 1.0.0
     */
    function saveReferral()
    {
        $key = (string)self::$input->post('option_key', NULL);
        $key = Validation::validateInputAlpha($key);
        $allow_save_referral = true;
        if (!Woocommerce::hasAdminPrivilege()) {
            $allow_save_referral = false;
        }
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        $response = array();
        if ($allow_save_referral && !empty($key) && Woocommerce::verify_nonce($wlpr_none,'wlpr-referral-none')) {
            $data = self::$input->post();
            if (isset($data['option_key'])) {
                unset($data['option_key']);
            }
            if (isset($data['action'])) {
                unset($data['action']);
            }
            $data = apply_filters('wlpr_point_before_save_referral', $data, $key);
            $validate_data = Validation::validateReferralTab($_REQUEST);
            if(is_array($validate_data)){
                foreach ($validate_data as $field => $messages){
                    $validate_data[$field] = str_replace('Wlpr','',implode(',',$messages));
                }
                $response['error'] = true;
                $response['field_error'] = $validate_data;
                $response['message'] = __('Referral not saved!');
            }
            if(!isset($response['error']) || !$response['error']){
                foreach ($data as $d_key => $d_value){
                    if(in_array($d_key,array(
                        'wlpr_cart_referee_reward_message',
                        'wlpr_before_referral_message',
                        'wlpr_referrer_message',
                        'wlpr_referee_message',
                        'wlpr_referee_discount_message',
                        'wlpr_share_subject',
                        'wlpr_share_content'
                    ))){
                        $d_value = stripslashes($d_value);
                        $data[$d_key] = Woocommerce::getCleanHtml($d_value);
                    }
                }
                update_option($key, $data, true);
                do_action('wlpr_point_after_save_referral', $data, $key);
                $response['error'] = false;
                $response['message'] = esc_html__('Referral saved successfully!', WLPR_TEXT_DOMAIN);
            }

        } else {
            $response['error'] = true;
            $response['message'] = esc_html__('Referral data not saved!', WLPR_TEXT_DOMAIN);
        }
        wp_send_json($response);
    }

    /**
     * Update expire date
     * @throws Exception
     */
    function update_expire_date(){
        $action_id = (int)self::$input->post('action_id', '');
        $expire_date = (string)self::$input->post('expire_date', '');
        $wlpr_none = (string)self::$input->post('wlpr_none', '');
        $allow_update = true;
        if (!Woocommerce::hasAdminPrivilege()) {
            $allow_update = false;
        }
        if($allow_update && !empty($action_id) && !empty($expire_date) && Woocommerce::verify_nonce($wlpr_none,'wlpr-update-expire-date-none')){
            $expire_date = \Wlpr\App\Helpers\Loyalty::woocommerce()->convert_wp_time_to_utc($expire_date, 'Y-m-d H:i:s');
            $point_action = new PointAction();
            $point_action->updateRow(array(
                'expire_date' => $expire_date
            ), array(
                'id' => $action_id
            ));
            $response['error'] = false;
            $response['message'] = esc_html__('Expire date updated!', WLPR_TEXT_DOMAIN);
        }else{
            $response['error'] = true;
            $response['message'] = esc_html__('Expire date not found!', WLPR_TEXT_DOMAIN);
        }
        wp_send_json($response);
    }

    /**
     * Validate customize launcher settings
     * @param $error
     * @param $unsanitized_value
     * @param $setting
     * @return \WP_Error
     */
    function validateCustomizeSettings($error,$unsanitized_value,$setting){
        //text area validation
        if(isset($setting->id) && in_array($setting->id,
                array(
                    'wlpr_launcher_login_header_content',
                    'wlpr_launcher_login_referee_content',
                    'wlpr_launcher_body_login_referee_content',
                    'wlpr_launcher_login_no_referral_content',
                    'wlpr_launcher_login_referral_content',
                    'wlpr_launcher_login_review_content',
                    'wlpr_launcher_guest_header_content',
                    'wlpr_launcher_guest_signup_content',
                    'wlpr_launcher_guest_review_content'
                )
            )){
            $new_value = Woocommerce::getCleanHtml($unsanitized_value);
            if($unsanitized_value != $new_value){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }elseif(isset($setting->id) && in_array($setting->id,array('wlpr_launcher[preview_customer]'))){
            //email validation
            if(!($unsanitized_value == 0 || filter_var($unsanitized_value, \FILTER_VALIDATE_EMAIL) !== false)){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }elseif(isset($setting->id) && in_array($setting->id,array('wlpr_launcher_login_referee_card_enable',
                'wlpr_launcher_login_referral_card_enable',
                'wlpr_launcher_login_review_card_enable',
                'wlpr_launcher_guest_signup_card_enable',
                'wlpr_launcher_guest_review_card_enable'
            ))){
            // yes, no
            $acceptable = array('yes', 'no', 1, '1', 0, '0');
            if(!in_array($unsanitized_value, $acceptable, true)){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }else if(isset($setting->id) && in_array($setting->id,array('wlpr_launcher_login_referee_ordering',
                'wlpr_launcher_login_referral_ordering',
                'wlpr_launcher_login_review_ordering',
                ))){
            //Numeric
            if(!is_numeric($unsanitized_value)){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }else if(isset($setting->id) && in_array($setting->id,array('wlpr_launcher_login_referee_heading',
                'wlpr_launcher_login_referral_heading',
                'wlpr_launcher_login_review_heading',
                'wlpr_launcher_guest_signup_heading',
                'wlpr_launcher_guest_signup_button_text',
                'wlpr_launcher_guest_signup_signin_link_text',
                'wlpr_launcher_guest_review_heading',
                'wlpr_launcher_font_family'
                ))){
            // Alpha numeric with space
            if(!preg_match('/^([a-z0-9 ])+$/i', $unsanitized_value)){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }else if(isset($setting->id) && in_array($setting->id,array('wlpr_launcher_login_referee_background_color',
                'wlpr_launcher_login_referee_text_color',
                'wlpr_launcher_login_referral_background_color',
                'wlpr_launcher_login_referral_text_color',
                'wlpr_launcher_login_review_background_color',
                'wlpr_launcher_login_review_text_color',
                'wlpr_launcher_guest_signup_button_color',
                'wlpr_launcher_guest_signup_button_text_color',
                'wlpr_launcher_guest_signup_text_color',
                'wlpr_launcher_guest_signup_background_color',
                'wlpr_launcher_guest_review_background_color',
                'wlpr_launcher_guest_review_text_color',
                'wlpr_launcher_header_background_color',
                'wlpr_launcher_header_text_color',
                'wlpr_launcher_body_background_color'
            ))){
            // color
            $color = (bool)preg_match('/^#(([0-9a-fA-F]{2}){3}|([0-9a-fA-F]){3})$/', $unsanitized_value);
            if(!$color){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }else if(isset($setting->id)&& !empty($unsanitized_value) && in_array($setting->id,array('wlpr_launcher_guest_signup_signin_prefix_text',
                'wlpr_launcher_guest_signup_button_url',
                'wlpr_launcher_guest_signup_signin_url',
                'wlpr_launcher[custom_css]'
            ))){
            // alpha numeric with special charactor
            //$special = (bool)preg_match("/^[A-Za-z0-9_! \"#$%&'()*+{},\-.\\:\/;=?@^_]+$/", $unsanitized_value);
            $special = (bool)preg_match("/^[A-Za-z0-9_! \"#$%&'()*+{},\s\-.\\:\/;=?@^_]+$/", $unsanitized_value);
            if(!$special){
                $error = new \WP_Error( 'invalid_value', __( 'Invalid value.',WLPR_TEXT_DOMAIN ) );
            }
        }
        return $error;
    }
}