<?php
class WC_Atome_Gateway extends WC_Payment_Gateway
{

  //when update a version, should update this version and above
  const VERSION = '2.5.11';
  const BASE_URL = "https://gateway.apaylater.com";

  /**
   * Whether or not logging is enabled
   *
   * @var bool
   */
  public static $log_enabled = false;

  /**
   * Logger instance
   *
   * @var WC_Logger
   */
  public static $log = false;

  private static $instance = null;

  public function __construct()
  {
    $this->id = 'atome';
    $this->icon = self::BASE_URL . '/plugins/common/assets/svg/logo-payments.svg';
    $this->has_fields = false;
    $this->method_title = 'Atome';
    $this->title = 'Atome';
    $this->method_description = 'Atome Payment Gateway';
    $this->supports = array(
      'products',
      'refunds'
    );

    $this->localeInfo = array();
    $this->default_language_contents = $this->get_language_contents('English');
    $this->current_language_contents = array();

    $this->init_form_fields();
    $this->init_settings();

    self::$log_enabled = 'staging' == $this->get_option('api_environment') || 'yes' == $this->get_option('debug_mode');

    $this->description = $this->get_description();

    $this->siteUrl = get_site_url();

    add_action('woocommerce_api_atome', array($this, 'callback_handler'));
    add_action('woocommerce_api_result', array($this, 'result_handler'));
    add_action('woocommerce_api_link', array($this, 'link_handler'));
    add_action('woocommerce_api_query', array($this, 'query_handler'));
    add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    add_action('woocommerce_create_order', array($this, 'atome_create_quote'), 10, 2);
    add_filter('woocommerce_new_order_data', array($this, 'filter_woocommerce_new_order_data'), 10, 1);
    add_filter('woocommerce_available_payment_gateways', array($this, 'check_cart_within_limits'), 10, 1);
  }


  /**
   * Logging method.
   *
   * @param string $message Log message.
   * @param string $level Optional. Default 'info'. Possible values:
   *                      emergency|alert|critical|error|warning|notice|info|debug.
   */
  public static function log($message, $level = 'info')
  {
    if (self::$log_enabled) {
      if (empty(self::$log)) {
        self::$log = wc_get_logger();
      }

      self::$log->log($level, $message, array('source' => 'atome'));
    }
  }

  public static function add_log($message, $level = 'info')
  {
    if (empty(self::$log)) {
      self::$log = wc_get_logger();
    }

    self::$log->log($level, $message, array('source' => 'atome'));
  }

  public function get_api_base_url()
  {
    $apiBaseUrls = array(
      'staging'    => 'https://api.apaylater.net/v1',
      'production' => 'https://api.apaylater.com/v1',
    );

    $env = $this->get_option('api_environment');

    return isset($apiBaseUrls[$env]) ? $apiBaseUrls[$env] . '/' : 'https://api.apaylater.com/v1/';
  }

  private function get_config($name, $defaultValue = '')
  {
    $last_updated_time = $this->get_option('last_updated_time');

    if (empty($last_updated_time) || (time() - 3600) > $last_updated_time) {
      $configs = $this->get_configs_from_atome();
      if (!empty($configs)) {
        foreach ($configs as $key => $value) {
          $this->update_option($key, $value);
        }

        $this->update_option('last_updated_time', time());
      } else {
        $this->update_option($name, ''); // empty value when getting from Atome is error
      }
    }

    $result = $this->get_option($name);

    if ('' !== $result) {
      return $result;
    }

    $localeInfo = $this->get_current_locale_info();

    return isset($localeInfo[$name]) ? $localeInfo[$name] : $defaultValue;
  }

  private function get_configs_from_atome()
  {
    $url = $this->get_api_base_url() . 'variables/' . strtoupper($this->get_option('country'));

    try {
      $response = wp_remote_get($url, array(
        'method' => 'GET',
      ));

      if (is_wp_error($response)) {
        throw new Exception('get configs info from atome error, url: ' . $url, -1);
      }

      if (empty($response['body'])) {
        throw new Exception('get configs info from atome error: Empty body.', -1);
      }

      $this->log('get configs info from atome response: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));

      $body = json_decode($response['body'], true);

      if (!is_array($body) || empty($body) || !isset($body['country'])) {
        return [];
      }

      foreach ($body as $key => $value) {
        if (in_array($key, ['country'])) {
          unset($body[$key]);
        }

        if ('minSpend' == $key) {
          $localeInfo = $this->get_current_locale_info();
          $intFactor  = isset($this->localeInfo['int_factor']) ? $this->localeInfo['int_factor'] : 100;
          $body['minimum_spend'] = intval($value / $intFactor);
          unset($body[$key]);
        }
      }

      return $body;
    } catch (Exception $e) {

      $this->log('get configs info from atome exception: ' . wc_print_r($e, true));
    }

    return [];
  }

  public static function get_instance()
  {
    if (self::$instance == null) {
      self::$instance = new self;
    }
    return self::$instance;
  }

  public function process_admin_options()
  {
    if (version_compare(WC()->version, '3.6.0', '<')) {
      WC_Admin_Settings::add_error(__('Sorry, Atome plugin does not support WooCommerce version below 3.6. To continue, kindly upgrade your WooCommerce version.', 'woocommerce'));
      return;
    }
    $country =  $this->get_post_data()['woocommerce_' . $this->id . '_country'];
    $language = $this->get_post_data()['woocommerce_' . $this->id . '_language'];
    if ($language == 'zh' && !in_array($country, ['hk', 'tw'])) {
      WC_Admin_Settings::add_error('Chinese(traditional) is not supported in this area, please config again');
      return;
    }
    parent::process_admin_options();
    $api_key = $this->get_post_data()['woocommerce_' . $this->id . '_api_key'];
    $password = $this->get_post_data()['woocommerce_' . $this->id . '_password'];
    $basic = base64_encode($api_key . ':' . $password);
    $header = array(
      'Authorization' => 'basic' . $basic,
      'Content-Type' => 'application/json',
    );

    $args = array(
      'method' => 'POST',
      'headers' => $header,
      'body' => json_encode(array(
        'callbackUrl' => get_site_url() . '/?wc-api=LINK',
        'countryCode' => strtoupper($country)
      ))
    );

    $response = wp_remote_post($this->get_api_base_url() . 'auth', $args);

    if (is_wp_error($response)) {
      $this->add_log('process_admin_options Reponse: ' . json_encode($response));
      WC_Admin_Settings::add_error(__('This is general notice', 'woocommerce'));
    } else {
      $this->add_log('process_admin_options Reponse: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));
      $body = json_decode($response['body'], true);
      if ($body['code'] == 'SUCCESS') {
        // WC_Admin_Settings::add_message(__('Your settings have been saved', 'woocommerce'));
      } else {
        WC_Admin_Settings::add_error(__($body['message'], 'woocommerce'));
      }
    }
  }

  function invalid_key_notice()
  {
    $class = 'notice notice-error';
    $message = __('Invalid key, please try again', 'sample-text-domain');

    printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), esc_html($message));
  }

  function get_header()
  {
    $api_key = $this->get_option('api_key');
    $password = $this->get_option('password');
    $basic = base64_encode($api_key . ':' . $password);
    return array(
      'Authorization' => 'basic' . $basic,
      'Content-Type' => 'application/json',
      'X-Version' => self::VERSION,
    );
  }

  function get_description()
  {
    $localeInfo       = $this->get_current_locale_info();
    $language         = $this->get_option('language');
    $minimum_spend    = $this->get_config('minimum_spend');

    $description = preg_replace_callback('/{{(\w+?)}}/', function ($matches) use ($localeInfo, $language, $minimum_spend) {

      if ('currency_name' == $matches[1]) {
        return isset($localeInfo['currency_name'][$language]) ? $localeInfo['currency_name'][$language] : 'Singapore Dollar';
      } else if ('minimum_spend' == $matches[1]) {
        return wc_price($minimum_spend);
      }

      return isset($localeInfo[$matches[1]]) ? $localeInfo[$matches[1]] : '';
    }, $this->get_messages('description'));

    if (preg_match('/otobodycare.com/', get_option('siteurl'))) {
      $description = "<img src='" . self::BASE_URL . "/plugins/common/assets/svg/logo-pd.svg' alt='Atome' style='position: relative; left: 20px; height: 20px;'>\n\n" . $description;
    }

    return $description;
  }

  function get_current_locale_info()
  {
    if (empty($this->localeInfo)) {
      $localeInfoPath = WOO_ATOME_PLUGIN_PATH . '/i18n/locale-info.php';

      if (file_exists($localeInfoPath)) {
        $localeInfoArr = include($localeInfoPath);
      }
      $this->log('[load_current_locale_info]: localeInfoArr => ' . json_encode($localeInfoArr));

      $country          = $this->get_option('country');
      $this->log('[load_current_locale_info]:Current country => ' . $country);

      $this->localeInfo = isset($localeInfoArr[$country]) ? $localeInfoArr[$country] : array();
      $this->log('[get_current_locale_info]: currentLocaleInfo => ' . json_encode($this->localeInfo));
    }

    return $this->localeInfo;
  }

  function get_language_contents($languageFileName = 'English')
  {

    $path = WOO_ATOME_PLUGIN_PATH . '/i18n/languages/' . $languageFileName . '.php';

    if (!file_exists($path)) {
      $path = WOO_ATOME_PLUGIN_PATH . '/i18n/languages/English.php';
    }

    $languageContents = require $path;

    return $languageContents;
  }

  function check_currency()
  {
    $wcCurrency    = get_woocommerce_currency();
    $localeInfo    = $this->get_current_locale_info();
    $atomeCurrency = isset($localeInfo['currency_code']) ? $localeInfo['currency_code'] : '';

    $this->log('[check_currency_info]: wcCurrency => ' . $wcCurrency . ', atomeCurrency => ' . $atomeCurrency);

    return !empty($wcCurrency) && !empty($atomeCurrency) && $wcCurrency == $atomeCurrency;
  }

  function get_messages($msgKeys = array())
  {
    if (empty($this->current_language_contents)) {
      $localeInfo       = $this->get_current_locale_info();
      $language         = $this->get_option('language');
      $languageFileName = isset($localeInfo['language_file'][$language]) ? $localeInfo['language_file'][$language] : 'English';
      $this->current_language_contents = array_merge($this->default_language_contents, $this->get_language_contents($languageFileName));
    }

    if (is_array($msgKeys)) {
      $result = array();

      foreach ($msgKeys as $v) {
        $result[$v] = isset($this->current_language_contents[$v]) ? $this->current_language_contents[$v] : '';
      }

      return $result;
    }

    return isset($this->current_language_contents[$msgKeys]) ? $this->current_language_contents[$msgKeys] : '';
  }

  public function check_display_atome_order_number()
  {
    return 'yes' === $this->get_option('display_atome_order_number');
  }

  function check_api_auth($str)
  {
    $this->log('check_api_auth: ' . $str);
    try {
      $str = base64_decode(substr($str, 5));
      $arr = explode(':', $str);
      $is_auth = $arr[0] === $this->get_option('api_key') && $arr[1] === $this->get_option('password');
      if (!$is_auth) {
        header("HTTP/1.1 401 Unauthorized");
        die(json_encode(array('code' => '401', 'message' => 'You are unauthorized to access')));
      }
    } catch (Exception $e) {
      header("HTTP/1.1 401 Unauthorized");
      die(json_encode(array('code' => '401', 'message' => 'You are unauthorized to access')));
    }
  }
  /**
   * Plugin options, we deal with it in Step 3 too
   */
  public function init_form_fields()
  {

    $this->form_fields = array(
      'basic_settings' => array('title' => 'Basic settings', 'type' => 'title', 'description' => ''),
      'country' => array(
        'title'   => 'Country/Region',
        'type'    => 'select',
        'options' => array('sg' => 'Singapore', 'hk' => 'Hong Kong', 'my' => 'Malaysia', 'id' => 'Indonesia', 'vn' => 'Vietnam', 'th' => 'Thailand', 'tw' => 'Taiwan', 'ph' => 'Philippines'),
        'default' => 'sg',
      ),
      'language' => array(
        'title'   => 'Language',
        'type'    => 'select',
        'options' => array('en' => 'English', 'zh' => 'Chinese(traditional)', 'id' => 'Bahasa', 'vi' => 'Vietnamese', 'th' => 'Thai'),
        'default' => 'en',
      ),
      'api_environment' => array(
        'title'   => 'Environment',
        'type'    => 'select',
        'options' => array('production' => 'Production', 'staging' => 'Staging'),
        'default' => 'production',
      ),
      'api_key' => array(
        'title'       => 'ApiKey',
        'type'        => 'text',
        'description' => 'For access auth to atome payment gateway',
        'desc_tip'    => true,
      ),
      'password' => array(
        'title'       => 'Password',
        'type'        => 'password',
        'description' => 'For access auth to atome payment gateway',
        'desc_tip'    => true,
      ),
      'sku_permission' => array(
        'title'   => 'SKU Permission',
        'type'    => 'checkbox',
        'label'   => 'Allow Atome to display SKU information in our APP to increase exposure',
        'default' => 'yes',
      ),
      'price_divider_settings' => array('title' => 'Price divider settings', 'type' => 'title', 'description' => ''),
      'price_divider' => array(
        'title'   => '[Atome] Price Divider',
        'type'    => 'checkbox',
        'label'   => 'Show instalment price of products to increase conversion rate',
        'default' => 'yes'
      ),
      'price_divider_created_by' => array(
        'title'   => '[Atome] Price Divider Created By',
        'type'    => 'select',
        'label'   => 'Which type the price divider created by',
        'options' => array('plugin' => 'Plugin', 'script' => 'Script'),
        'default' => 'plugin',
      ),
      'price_divider_applied_on' => array(
        'title'   => '[Atome] Price Divider Applied On',
        'type'    => 'select',
        'label'   => 'Which pages the price divider applied on',
        'options' => array('all' => 'All pages', 'product' => 'Only product page'),
        'default' => 'all',
      ),
      'price_divider_position' => array(
        'title'   => '[Atome] Price Divider Position',
        'type'    => 'select',
        'label'   => 'Where the price divider is showing',
        'options' => array('below_product_price' => 'Below Product Price', 'above_add_to_cart' => 'Above Add to cart', 'below_add_to_cart' => 'Below Add to cart'),
        'default' => 'below_product_price',
      ),
      'advanced_settings' => array('title' => 'Advanced settings', 'type' => 'title', 'description' => ''),
      'order_created_stage' => array(
        'title'   => 'Order Created',
        'type'    => 'select',
        'options' => array('before_payment' => 'Before payment', 'after_payment' => 'After paid successfully'),
        'default' => 'after_payment',
      ),
      'max_spend' => array(
        'title'   => 'Max Spend',
        'type'        => 'Decimal',
        'description' => 'Order will not be placed when order total is over than max spend. Price divider will not be shown when product price is over than max spend.',
        'placeholder' => '0 or empty for no limit',
        'desc_tip'    => true,
        'default'     => 0,
      ),
      'debug_mode' => array(
        'title'   => 'Debug Mode',
        'label'   => 'Debug mode',
        'type'    => 'checkbox',
        'default' => 'no',
      ),
      'display_atome_order_number' => array(
        'title'   => 'Display atome order number',
        'label'   => 'Display the order number which is the same with Atome side in one column in woocommerce order list',
        'type'    => 'checkbox',
        'default' => 'no',
      ),
    );
  }

  /**
   * You will need it if you want your custom credit card form, Step 4 is about it
   */
  public function payment_fields()
  {
    echo wpautop(wptexturize($this->description));
  }

  /*
		 * Custom CSS and JS, in most cases required only when you decided to go with a custom credit card form
		 */
  public function payment_scripts()
  {
  }

  /*
 		 * Fields validation, more in Step 5
		 */
  public function validate_fields()
  {
  }

  // step 1
  public function atome_create_quote($order, $checkout)
  {
    $data = $checkout->get_posted_data();

    $this->add_log('[atome_create_quote]: posted_data => ' . json_encode($data) . ', $_POST => ' . json_encode($_POST));

    if ($data['payment_method'] != $this->id) {
      return;
    }

    if ($this->get_option('order_created_stage') === 'before_payment') {
      $this->add_log('[atome_create_quote]: order_created_stage => before_payment, return');
      return;
    }

    global $wpdb;
    $wpdb->query('START TRANSACTION');
    try {
      $post_id = wp_insert_post(array(
        'post_content' => 'Redirecting to Atome to complete payment...',
        'post_title' => 'Atome Order',
        'post_status' => 'publish',
        'post_type' => 'atome_quote',
        'post_name' => 'atome-order-' . time() . rand(100, 999),
      ), true);

      if (is_wp_error($post_id)) {
        $this->add_log('[atome_create_quote] insert post error: ' . wc_print_r($post_id, true));
        throw new Exception('Insert post error', -1001);
      }

      $cart = WC()->cart;

      $this->add_log('[atome_create_quote] -> cart:' . json_encode($cart));

      $cart_hash = $cart->get_cart_hash();
      $available_gateways = WC()->payment_gateways->get_available_payment_gateways();

      $chosen_shipping_methods = WC()->session->get('chosen_shipping_methods');
      $shipping_packages = WC()->shipping()->get_packages();

      $customer_id = apply_filters('woocommerce_checkout_customer_id', get_current_user_id());
      $order_vat_exempt = ($cart->get_customer()->get_is_vat_exempt() ? 'yes' : 'no');
      $currency = get_woocommerce_currency();
      $prices_include_tax = (get_option('woocommerce_prices_include_tax') === 'yes');
      $customer_ip_address = WC_Geolocation::get_ip_address();
      $customer_user_agent = wc_get_user_agent();
      $customer_note = (isset($data['order_comments']) ? $data['order_comments'] : '');
      $payment_method = (isset($available_gateways[$data['payment_method']]) ? $available_gateways[$data['payment_method']] : $data['payment_method']);
      $shipping_total = $cart->get_shipping_total();
      $discount_total = $cart->get_discount_total();
      $discount_tax = $cart->get_discount_tax();
      $cart_tax = $cart->get_cart_contents_tax() + $cart->get_fee_tax();
      $shipping_tax = $cart->get_shipping_tax();
      $total = $cart->get_total('edit');


      add_post_meta($post_id, 'posted', $this->special_encode($data));
      add_post_meta($post_id, 'cart', $this->special_encode($cart));

      add_post_meta($post_id, 'cart_hash', $this->special_encode($cart_hash));

      add_post_meta($post_id, 'chosen_shipping_methods', $this->special_encode($chosen_shipping_methods));
      add_post_meta($post_id, 'shipping_packages', $this->special_encode($shipping_packages));

      add_post_meta($post_id, 'customer_id', $this->special_encode($customer_id));
      add_post_meta($post_id, 'order_vat_exempt', $this->special_encode($order_vat_exempt));
      add_post_meta($post_id, 'currency', $this->special_encode($currency));
      add_post_meta($post_id, 'prices_include_tax', $this->special_encode($prices_include_tax));
      add_post_meta($post_id, 'customer_ip_address', $this->special_encode($customer_ip_address));
      add_post_meta($post_id, 'customer_user_agent', $this->special_encode($customer_user_agent));
      add_post_meta($post_id, 'customer_note', $this->special_encode($customer_note));
      add_post_meta($post_id, 'payment_method', $this->special_encode($payment_method));
      add_post_meta($post_id, 'shipping_total', $this->special_encode($shipping_total));
      add_post_meta($post_id, 'discount_total', $this->special_encode($discount_total));
      add_post_meta($post_id, 'discount_tax', $this->special_encode($discount_tax));
      add_post_meta($post_id, 'cart_tax', $this->special_encode($cart_tax));
      add_post_meta($post_id, 'shipping_tax', $this->special_encode($shipping_tax));
      add_post_meta($post_id, 'total', $this->special_encode($total));
      //support other plugins
      add_post_meta($post_id, 'all_post_data', $this->special_encode($_POST));
      //support Proword/PW Woocommerce Advanced Gift Rules
      $session_data_for_pw_gift = WC()->session->get('group_order_data');
      add_post_meta($post_id, 'session_data_for_pw_gift', $this->special_encode($session_data_for_pw_gift));
      $this->add_log('[atome_create_quote]: session_data_for_pw_gift => ' . wc_print_r($session_data_for_pw_gift, true));

      $result = $this->create_payment($post_id);

      $wpdb->query('COMMIT');

      if (is_ajax()) {
        wp_send_json($result);
      } else {
        wp_redirect($result['redirect']);
      }
      exit;
    } catch (Exception $e) {
      $wpdb->query('ROLLBACK');
      $errorMsg = $e->getMessage() . '(' . $e->getCode() . ')';
      $this->add_log('[atome_create_quote] exception: ' . $errorMsg);
      $this->checkoutErrmsg($errorMsg);
      return -1;
    }
  }

  // step 2
  public function atome_create_wc_order($post_id)
  {
    $checkout = WC()->checkout;
    $data = $this->special_decode(get_post_meta($post_id, 'posted', true));
    $cart = $this->special_decode(get_post_meta($post_id, 'cart', true));

    if (!$cart instanceof WC_Cart) {
      throw new Exception('invalid WC_Cart instance, cart: ' . wc_print_r($cart, true));
    }

    WC()->cart = $cart;

    $cart_hash = $this->special_decode(get_post_meta($post_id, 'cart_hash', true));

    $chosen_shipping_methods = $this->special_decode(get_post_meta($post_id, 'chosen_shipping_methods', true));
    $shipping_packages = $this->special_decode(get_post_meta($post_id, 'shipping_packages', true));

    $customer_id = $this->special_decode(get_post_meta($post_id, 'customer_id', true));
    $order_vat_exempt = $this->special_decode(get_post_meta($post_id, 'order_vat_exempt', true));
    $currency = $this->special_decode(get_post_meta($post_id, 'currency', true));
    $prices_include_tax = $this->special_decode(get_post_meta($post_id, 'prices_include_tax', true));
    $customer_ip_address = $this->special_decode(get_post_meta($post_id, 'customer_ip_address', true));
    $customer_user_agent = $this->special_decode(get_post_meta($post_id, 'customer_user_agent', true));
    $customer_note = $this->special_decode(get_post_meta($post_id, 'customer_note', true));
    $payment_method = $this->special_decode(get_post_meta($post_id, 'payment_method', true));
    $shipping_total = $this->special_decode(get_post_meta($post_id, 'shipping_total', true));
    $discount_total = $this->special_decode(get_post_meta($post_id, 'discount_total', true));
    $discount_tax = $this->special_decode(get_post_meta($post_id, 'discount_tax', true));
    $cart_tax = $this->special_decode(get_post_meta($post_id, 'cart_tax', true));
    $shipping_tax = $this->special_decode(get_post_meta($post_id, 'shipping_tax', true));
    $total = $this->special_decode(get_post_meta($post_id, 'total', true));
    $all_post_data = $this->special_decode(get_post_meta($post_id, 'all_post_data', true));
    $session_data_for_pw_gift = $this->special_decode(get_post_meta($post_id, 'session_data_for_pw_gift', true));

    $this->add_log('atome_create_wc_order: posted_data => ' . json_encode($data) . ', all_post_data => ' . json_encode($all_post_data) . ', session_data_for_pw_gift => ' . wc_print_r($session_data_for_pw_gift, true));

    //support other plugins
    if (!empty($all_post_data) && is_array($all_post_data)) {
      $_POST = array_merge($_POST, $all_post_data);
    }

    $this->add_log('atome_create_wc_order: _POST => ' . json_encode($_POST));

    if (!empty($session_data_for_pw_gift)) {
      WC()->session->set('group_order_data', $session_data_for_pw_gift);
    }

    try {
      wp_delete_post($post_id, true);
      $order = new WC_Order();

      $fields_prefix = array(
        'shipping' => true,
        'billing'  => true,
      );

      $shipping_fields = array(
        'shipping_method' => true,
        'shipping_total'  => true,
        'shipping_tax'    => true,
      );

      foreach ($data as $key => $value) {
        if (is_callable(array($order, "set_{$key}"))) {
          $order->{"set_{$key}"}($value);
        } elseif (isset($fields_prefix[current(explode('_', $key))])) {
          if (!isset($shipping_fields[$key])) {
            $order->update_meta_data('_' . $key, $value);
          }
        }
      }

      if (isset($_POST['shipping_phone']) && (!isset($data['shipping_phone']) || empty($data['shipping_phone']))) {
        $data['shipping_phone'] = $_POST['shipping_phone'];
      }

      $this->add_log('atome_create_wc_order: posted_data_after_POST => ' . json_encode($data));

      $order->set_cart_hash($cart_hash);
      $order->set_customer_id($customer_id);
      $order->add_meta_data('is_vat_exempt', $order_vat_exempt);
      isset($data['shipping_phone']) && $order->add_meta_data('shipping_phone', $data['shipping_phone']);
      $order->set_currency($currency);
      $order->set_prices_include_tax($prices_include_tax);
      $order->set_customer_ip_address($customer_ip_address);
      $order->set_customer_user_agent($customer_user_agent);
      $order->set_customer_note($customer_note);
      $order->set_payment_method($payment_method);
      $order->set_shipping_total($shipping_total);
      $order->set_discount_total($discount_total);
      $order->set_discount_tax($discount_tax);
      $order->set_cart_tax($cart_tax);
      $order->set_shipping_tax($shipping_tax);
      $order->set_total($total);

      $checkout->create_order_line_items($order, $cart);
      $checkout->create_order_fee_lines($order, $cart);
      $checkout->create_order_shipping_lines($order, $chosen_shipping_methods, $shipping_packages);
      $checkout->create_order_tax_lines($order, $cart);
      $checkout->create_order_coupon_lines($order, $cart);

      $GLOBALS['atome_quote_id'] = $post_id;

      do_action('woocommerce_checkout_create_order', $order, $data);

      $order_id = $order->save();

      if (!$order_id) {
        throw new Exception('create order failed!');
      }

      do_action('woocommerce_checkout_update_order_meta', $order_id, $data);

      if (isset($GLOBALS['atome_quote_id'])) {
        unset($GLOBALS['atome_quote_id']);
      }

      do_action('woocommerce_checkout_order_processed', $order_id, $data, $order);

      return $order;
    } catch (Exception $e) {
      $this->add_log('Atome Create Wc Order Exception: ', wc_print_r($e, true));

      throw $e;
    }
  }

  private function get_data($post_id)
  {
    try {
      $data = $this->special_decode(get_post_meta($post_id, 'posted', true));
      $cart = WC()->cart;

      $shipping_total = $this->special_decode(get_post_meta($post_id, 'shipping_total', true));
      $cart_tax = $this->special_decode(get_post_meta($post_id, 'cart_tax', true));

      $cart_items_arr = $cart->get_cart();
      $items = array();
      foreach ($cart_items_arr as $cart_item_key => $values) {
        $product = $values['data'];
        $items[] = array(
          'name' => $product->get_name(),
          'itemId' => $product->get_id(),
          'quantity' => $values['quantity'],
          'price' => $this->integerize($product->get_price()),
          'variationName' => $product->get_name(),
          'originalPrice' => $this->integerize($product->get_regular_price())
        );
      }

      $applied_coupons = $cart->get_applied_coupons();

      $data_all = array(
        'paymentCancelUrl' => wc_get_checkout_url(),
        'merchantReferenceId' => $post_id,
        'customerInfo' => array(
          'mobileNumber' => $data['billing_phone'],
          'fullName' =>  $data['billing_last_name'] . ' ' . $data['billing_first_name'],
          'email' => $data['billing_email']
        ),
        'shippingAddress' => array(
          'countryCode' => $data['shipping_country'],
          'lines' => array($data['shipping_city'], $data['shipping_address_1'], $data['shipping_address_2']),
          'postCode' => $data['shipping_postcode']
        ),
        'billingAddress' => array(
          'countryCode' => $data['billing_country'],
          'lines' => array($data['billing_city'], $data['billing_address_1'], $data['billing_address_2']),
          'postCode' => $data['billing_postcode']
        ),
        'taxAmount' => $this->integerize($cart_tax),
        'shippingAmount' => $this->integerize($shipping_total),
        'originalAmount' => $this->integerize($cart->get_subtotal()),
        'voucherCode' => isset($applied_coupons[0]) ? $applied_coupons[0] : '',
        'items' => $items,
      );

      return $data_all;
    } catch (Exception $e) {
      $this->add_log('get_data Exception: ' . wc_print_r($e, true));
      return array();
    }
  }

  private function special_encode($data)
  {
    return base64_encode(serialize($data));
  }

  private function special_decode($string)
  {
    return unserialize(base64_decode($string));
  }

  private function checkoutErrmsg($msg)
  {
    wp_send_json(array(
      'result'    => 'faiure',
      'messages'    => '<div class="woocommerce-error">' . $msg . '</div>'
    ));
  }

  private function minimum_spend_error()
  {
    $localeInfo             = $this->get_current_locale_info();
    $config_minimum_spend   = $this->get_config('minimum_spend', 10);
    $minimumSpendErrorStr   = preg_replace_callback('/{{(\w+?)}}/', function ($matches) use ($localeInfo, $config_minimum_spend) {

      if ('minimum_spend' == $matches[1]) {

        return wc_price($config_minimum_spend);
      }

      return isset($localeInfo[$matches[1]]) ? $localeInfo[$matches[1]] : '';
    }, $this->get_messages('minimum_spend_error'));

    return $minimumSpendErrorStr;
  }

  private function get_locale_text($text_key, $values = array())
  {
    $localeInfo  = $this->get_current_locale_info();
    $locale_text = preg_replace_callback('/{{(\w+?)}}/', function ($matches) use ($localeInfo, $values) {

      if (in_array($matches[1], array('max_spend'))) {
        return wc_price($values[$matches[1]]);
      }

      return isset($localeInfo[$matches[1]]) ? $localeInfo[$matches[1]] : '';
    }, $this->get_messages($text_key));

    return $locale_text;
  }

  public function filter_woocommerce_new_order_data($order_data)
  {
    if (array_key_exists('atome_quote_id', $GLOBALS) && is_numeric($GLOBALS['atome_quote_id']) && $GLOBALS['atome_quote_id'] > 0) {
      $order_data['import_id'] = (int) $GLOBALS['atome_quote_id'];
      unset($GLOBALS['atome_quote_id']);
    }
    return $order_data;
  }

  private function integerize($value)
  {
    $localeInfo = $this->get_current_locale_info();
    $intFactor  = isset($localeInfo['int_factor']) ? $localeInfo['int_factor'] : 100;

    $value *= $intFactor;
    $value = in_array($this->get_option('country'), ['id', 'vn']) ? ceil($value) : round($value);

    return intval($value);
  }

  private function get_origin_amount($sendAmount)
  {
    $localeInfo = $this->get_current_locale_info();
    $intFactor  = isset($localeInfo['int_factor']) ? $localeInfo['int_factor'] : 100;

    $amount = $sendAmount / $intFactor;

    return $amount;
  }

  public function format_price_for_price_divider($amount)
  {
    return in_array($this->get_option('country'), ['id', 'vn']) ? ceil($amount) : $amount;
  }

  private function check_minimum_spend($amount)
  {
    $config_minimum_spend = $this->get_config('minimum_spend', 10);
    $intFactor            = $this->get_config('int_factor', 100);
    $minimumSpend         = intval($config_minimum_spend * $intFactor);

    if ($amount < $minimumSpend) {
      return false;
    }

    return true;
  }

  private function check_max_spend($amount)
  {
    $max_spend        = $this->get_option('max_spend');
    $int_factor       = $this->get_config('int_factor', 100);
    $max_spend_amount = intval($max_spend * $int_factor);

    if (!empty($max_spend) && $max_spend_amount > 0 && $amount > $max_spend_amount) {
      return false;
    }

    return true;
  }

  /*
		 * We're processing the payments here, everything about it is in Step 5
		 */
  public function process_payment($order_id)
  {
    $this->add_log('atome_process_payment: order_id => ' . json_encode($order_id));
    $this->add_log('atome_process_payment: debug_backtrace => ' . json_encode(debug_backtrace()));

    $order = wc_get_order( $order_id );
    if (!$order instanceof WC_Order){
      $this->add_log('atome_process_payment valid order: false');
      return false;
    }

    if ($order->get_status() != 'pending') {
      $this->add_log('atome_process_payment wrong order status: '. $order->get_status());
      return false;
    }

    try {
      return $this->create_payment($order_id);
      exit;
    } catch (Exception $e) {
      $error_msg = $e->getMessage() . '(' . $e->getCode() . ')';
      $this->add_log('atome_process_payment: create_payment exception => ' . $error_msg);
      $this->checkoutErrmsg($error_msg);
      return false;
    }
  }

  private function create_payment($order_id)
  {
    $order_total = $this->get_order_total();
    $currency = get_woocommerce_currency();

    if (!$this->check_currency()) {
      throw new Exception('The currency is not supported or does not match, please try again.', -1011);
    }
    $amount = $this->integerize($order_total);
    if ('tw' == $this->get_option('country') && $amount != $order_total) {
      throw new Exception('The order total is not supported.', -1015);
    }

    if (!$this->check_minimum_spend($amount)) {
      throw new Exception($this->minimum_spend_error(), -1012);
    }

    if (!$this->check_max_spend($amount)) {
      throw new Exception($this->get_locale_text('max_spend_error', array('max_spend' => $this->get_option('max_spend'))), -1016);
    }

    $paymentResultUrl = get_site_url() . '/?wc-api=RESULT&token=##paymentToken';
    $callbackUrl = get_site_url() . '/?wc-api=ATOME&referenceId=' . $order_id;
    $requestBody = array(
      'referenceId' => $order_id,
      'currency' => $currency,
      'amount' => $amount,
      'callbackUrl' => $callbackUrl,
      'paymentResultUrl' => $paymentResultUrl,
    );
    $requestBody = array_merge($requestBody, $this->get_data($order_id));
    $args = array(
      'method' => 'POST',
      'headers' => $this->get_header(),
      'body' => json_encode($requestBody),
    );
    $apiUrl = $this->get_api_base_url() . 'payments';

    $this->add_log('Payment Request: ' . wc_print_r(array('url' => $apiUrl, 'args' => $args), true));

    $response = wp_remote_post($apiUrl, $args);

    if (is_wp_error($response)) {
      $this->add_log('Payment Response: ' . json_encode($response));
      throw new Exception('Connection error.', -1013);
    }

    $this->add_log('Payment Response: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));

    if (empty($response['body'])) {
      throw new Exception('Respnse error empty body.', -1014);
    }

    $body = json_decode($response['body'], true);
    if (isset($body['status']) && $body['status'] == 'PROCESSING') {
      $result = array(
        'result'   => 'success',
        'redirect' => $body['redirectUrl']
      );

      return $result;
    }

    throw new Exception('process payment failed, code:' . ($body['code'] ?? '') . ', message: ' . ($body['message'] ?? ''), -1019);
  }

  public function process_refund($order_id, $amount = null, $reason = '')
  {
    $requestBody = array(
      'refundAmount' => $this->integerize($amount)
    );
    if ('tw' == $this->get_option('country') && $amount != $requestBody['refundAmount']) {
      return new WP_Error('atome', 'The refund amount must be integer.');
    }

    $this->add_log('Refund Request Body: ' . wc_print_r($requestBody, true));

    $args = array(
      'method' => 'POST',
      'headers' => $this->get_header(),
      'body' => json_encode($requestBody),
    );

    $apiUrl = $this->get_api_base_url() . 'payments/' . $order_id . '/refund';

    $response = wp_remote_post($apiUrl, $args);
    
    if (is_wp_error($response)) {
      $this->add_log('Refund Response: ' . json_encode($response));
      return $response;
    }

    $this->add_log('Refund Response: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));
    
    if (empty($response['body'])) {
      return new WP_Error('atome-api', 'Empty Response');
    }

    $result = json_decode($response['body'], true);
    $this->add_log('Refund Result: ' . wc_print_r($result, true));

    if (isset($result['status']) && ($result['status'] == 'REFUNDED' || $result['status'] == 'PAID')) {
      return true;
    }

    return new WP_Error('atome-api', 'error: ' . (isset($result['code']) ? $result['code'] : '') . ' | ' . (isset($result['message']) ? $result['message'] : ''));
  }

  /*
		 * In case you need a webhook, like PayPal IPN etc
		 */
  public function callback_handler()
  {
    global $wpdb;
    $wpdb->query('START TRANSACTION');
    try {
      $order_id = $_GET['referenceId'];
      if (!$order_id) {
        throw new Exception('Can not get referenceId');
      }
      $args = array(
        'method' => 'GET',
        'headers' => $this->get_header(),
      );
      $apiUrl = $this->get_api_base_url() . 'payments/' . $order_id;
      $response = wp_remote_get($apiUrl, $args);

      if(is_wp_error($response)){
        $this->add_log( 'callback_handler Get Payment Info Response: '. json_encode($response));
        throw new Exception('get payment info error, url: '.$apiUrl, -1021);
      }

      $this->add_log( 'callback_handler Get Payment Info Response: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));

      if(empty($response['body'])){
          throw new Exception('get payment info error: Empty body.', -1022);
      }

      $body = json_decode($response['body'], true);

      if (!isset($body['status']) || $body['status'] != 'PAID') {
          throw new Exception('This order has not been paid: body => '.json_encode($body), -1029);
      }

      if ($this->get_option('order_created_stage') === 'before_payment') {
        $order = wc_get_order($order_id);
        if (!$order instanceof WC_Order) {
          throw new Exception('This order is not existed => order_id => ' . $order_id);
        }

        if ('pending' != $order->get_status()) {
          throw new Exception('This order has wrong status: status => ' . $order->get_status());
        }
      } else {
        $order = $this->atome_create_wc_order($order_id);
      }
      
      $order->update_status('processing');
      $order->payment_complete();
      wc_reduce_stock_levels($order_id);
      update_option('webhook_debug', $_POST);

      $wpdb->query('COMMIT');

      wp_send_json(array('code' => 'success'));
    } catch (Exception $e) {
      $wpdb->query('ROLLBACK');
      $this->add_log('Atome Callback Handler Exception: ' . wc_print_r($e, true));

      wp_send_json(array('code' => 'failed', 'message' => $e->getMessage()), 400);
    }
  }

  public function result_handler()
  {
    $token = $_GET['token'];

    $args = array(
      'method' => 'GET',
      'headers' => $this->get_header(),
    );
    $apiUrl = $this->get_api_base_url() . 'payments/?token=' . $token;
    $this->add_log('result_handler query payment Request: ' . wc_print_r(array('url' => $apiUrl, 'args' => $args), true));
    $response = wp_remote_get($apiUrl, $args);
    if (is_wp_error($response)) {
      $this->add_log('result_handler query payment Response: ' . json_encode($response));
      exit;
    }

    $this->add_log('result_handler query payment Response: ' . wc_print_r(array('body' => $response['body'], 'status' => $response['response']), true));
    $body = json_decode($response['body'], true);
    $order_id = $body['referenceId'];
    $order = wc_get_order($order_id);
    if ($order == false) {
      $this->add_log('result_handler query order empty, redirect to checkout.');
      wp_redirect(wc_get_checkout_url());
      die;
    }
    $this->add_log('result_handler query order success, redirect to thank you page.');
    if (wp_redirect($order->get_checkout_order_received_url())) {
      exit;
    }
  }

  public function link_handler()
  {
    $result = array('code' => 'success');

    switch ($_GET['check']) {
      case 'version':
        $result['version'] = self::VERSION;
        break;
      case 'query':
        foreach ($this->form_fields as $key => $item) {
          if (in_array($key, array('api_key', 'password'))) {
            continue;
          }
          $result[$key] = $this->get_option($key);
        }
        break;
      case 'plugins':
        $allHeaders = getallheaders();
        $authorization = isset($allHeaders['Authorization']) ? $allHeaders['Authorization'] : '';
        if (empty($authorization)) {
          $authorization = $_GET['Authorization'];
        }
        $this->check_api_auth($authorization);
        $result['plugins']        = get_plugins();
        $result['active_plugins'] = get_option('active_plugins', array());
        break;
      default:
        break;
    }

    wp_send_json($result);
  }

  public function query_handler()
  {
    ini_set('display_errors', 'Off');
    $allHeaders = getallheaders();
    $authorization = isset($allHeaders['Authorization']) ? $allHeaders['Authorization'] : '';
    if (empty($authorization)) {
      $authorization = $_GET['Authorization'];
    }
    $this->check_api_auth($authorization);
    function apply_atome_auth()
    {
      $gateways = WC_Atome_Gateway::get_instance();
      if ($gateways->get_option('sku_permission') === 'no') {
        return false;
      }
      return true;
    };
    add_filter('woocommerce_rest_check_permissions', 'apply_atome_auth', 10, 0);

    $queryUri = '/wc/v3';

    $type = 'products';
    $queryUri .= '/';
    $queryUri .= $type;

    $subType = isset($_GET['sub_type']) ? $_GET['sub_type'] : '';
    if (!empty($subType)) {
      $queryUri .= '/';
      $queryUri .= $subType;
    }

    if (isset($_GET['product_id'])) {
      $queryUri = $queryUri . '/' . $_GET['product_id'];
    } else if (isset($_GET['id'])) {
      $queryUri .= '/';
      $queryUri .= $_GET['id'];
    }

    $request = new WP_REST_Request('GET', $queryUri);
    $request->set_query_params($_GET);

    $response = rest_do_request($request);

    if ($response instanceof WP_REST_Response) {
      if ('200' != $response->get_status()) {
        $this->add_log('Query ' . $queryUri . ' response : ' . wc_print_r($response, true));
        wp_send_json(array(
          'headers' => $response->get_headers(),
          'status'  => $response->get_status(),
          'data'    => $response->get_data(),
          'links'   => $response->get_links(),
        ));
      }

      $rsp_data = $response->get_data();
      if (!is_array($rsp_data)) {
        wp_send_json(array('data' => wc_print_r($rsp_data, true)));
      }

      // products & price handling
      if ('' == $subType) {
        if (isset($rsp_data['id'])) {
          $rsp_data['price'] = wc_format_decimal($this->get_price_to_display($rsp_data['id']), wc_get_price_decimals());

          wp_send_json($rsp_data);
        }

        foreach ($rsp_data as &$product) {
          if (!isset($product['id'])) {
            continue;
          }
          $product['price'] = wc_format_decimal($this->get_price_to_display($product['id']), wc_get_price_decimals());
        }
      }

      wp_send_json($rsp_data);
    }

    $this->add_log('Query ' . $queryUri . ' response : ' . wc_print_r($response, true));
    wp_send_json(array(
      'error' => wc_print_r($response, true),
    ));
  }

  public function can_refund_order($order)
  {
    return $order->get_payment_method() === $this->id && $this->supports('refunds') && $order->get_remaining_refund_amount() != 0.00;
  }

  public function get_title()
  {
    if (is_admin() || wp_is_mobile()) {
      return 'Atome';
    }

    $localeInfo = $this->get_current_locale_info();

    return preg_replace_callback('/{{(\w+?)}}/', function ($matches) use ($localeInfo) {

      if ('atome_logo' == $matches[1]) {
        if (is_checkout() && get_query_var('wc-ajax') === 'update_order_review') {
          if (preg_match('/otobodycare.com/', get_option('siteurl'))) {
            return '';
          } else {
            return '<img class="atome-icon" src="' . self::BASE_URL . '/plugins/common/assets/svg/logo-pd.svg" alt="Atome" style="display: inline-block;vertical-align:text-bottom;float:none;width: initial;margin-left: initial;">';
          }
        } else {
          return 'Atome';
        }
      }

      return isset($localeInfo[$matches[1]]) ? $localeInfo[$matches[1]] : '';
    }, $this->get_messages('title'));
  }

  public function check_cart_within_limits($gateways)
  {
    if (is_admin()) {
      return $gateways;
    }

    if (!$this->check_currency()) {
      unset($gateways[$this->id]);
    }

    return $gateways;
  }

  public function get_icon()
  {
    $localeInfo = $this->get_current_locale_info();
    $result     = '';

    if (!empty($localeInfo['checkout_logo'])) {
      $checkout_icon_url = self::BASE_URL . '/plugins/common/assets/svg/' . $localeInfo['checkout_logo'];
      $result = '<img src=' . $checkout_icon_url . ' style="height:24px;display: inline-flex;vertical-align: middle;margin-left: 6px;width: initial;"/>' . $result;
    }
    return $result;
  }

  public function get_price_divider_logo_url()
  {
    return self::BASE_URL . '/plugins/common/assets/svg/logo-pd.svg';
  }

  public function get_price_to_display($product)
  {
    global $flycart_woo_discount_rules, $awdr_load_version;

    if (!$product instanceof WC_Product) {

      $product = wc_get_product($product);
    }

    $price_to_display = false;

    if ($awdr_load_version == "v2") {
      $custom_price = 0;
      if (method_exists($product, 'is_type') && $product->is_type('variable') && \Wdr\App\Controllers\DiscountCalculator::$config->getConfig('calculate_discount_from', 'sale_price')) {
        $custom_price = $product->get_variation_regular_price('min');
      }
      //not matched, apply_filters return false
      $price_to_display = apply_filters('advanced_woo_discount_rules_get_product_discount_price_from_custom_price', $price_to_display, $product, 1, $custom_price, 'discounted_price', false, false);
    } else if (isset($flycart_woo_discount_rules)) {
      try {
        $product_id = $product->get_id();
        if (isset($flycart_woo_discount_rules->pricingRules->products_has_discount[$product_id]) && $flycart_woo_discount_rules->pricingRules->products_has_discount[$product_id] == 1) {
          $price_to_display = $flycart_woo_discount_rules->pricingRules->getDiscountPriceForTheProduct($product);
        }
      } catch (Exception $e) {
        $this->log('Get Price From flycart_woo_discount_rules Exception: ' . wc_print_r($e, true));
      }
    }

    if ($price_to_display === false) {
      $price_to_display = wc_get_price_to_display($product);
    }

    return $price_to_display;
  }
}