<?php
/**
 * woobooking! Content Management System
 *
 * Copyright (C) 2005 - 2019 Open Source Matters, Inc. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace TourOperatorPro\Log;

defined('_TOUROPERATOR_EXEC') or die;
use TourOperatorPro\WbDate\WbDate;
/**
 * woobooking! Log Entry class
 *
 * This class is designed to hold log entries for either writing to an engine, or for
 * supported engines, retrieving lists and building in memory (PHP based) search operations.
 *
 
 */
class LogEntry
{
	/**
	 * Application responsible for log entry.
	 * @var    string
	 
	 */
	public $category;

	/**
	 * The message context.
	 *
	 * @var    array
	 * @since  3.8.0
	 */
	public $context;

	/**
	 * The date the message was logged.
	 * @var    Date
	 
	 */
	public $date;

	/**
	 * Message to be logged.
	 * @var    string
	 
	 */
	public $message;

	/**
	 * The priority of the message to be logged.
	 * @var    string
	 
	 * @see    LogEntry::$priorities
	 */
	public $priority = Log::INFO;

	/**
	 * List of available log priority levels [Based on the Syslog default levels].
	 * @var    array
	 
	 */
	protected $priorities = array(
		Log::EMERGENCY,
		Log::ALERT,
		Log::CRITICAL,
		Log::ERROR,
		Log::WARNING,
		Log::NOTICE,
		Log::INFO,
		Log::DEBUG,
	);

	/**
	 * Call stack and back trace of the logged call.
	 * @var    array
	 * @since  3.1.4
	 */
	public $callStack = array();

	/**
	 * Constructor
	 *
	 * @param   string  $message   The message to log.
	 * @param   int     $priority  Message priority based on {$this->priorities}.
	 * @param   string  $category  Type of entry
	 * @param   string  $date      Date of entry (defaults to now if not specified or blank)
	 * @param   array   $context   An optional array with additional message context.
	 *
	 * @since   1.7.0
	 */
	public function __construct($message, $priority = Log::INFO, $category = '', $date = null, array $context = array())
	{
		$this->message = (string) $message;

		// Sanitize the priority.
		if (!in_array($priority, $this->priorities, true))
		{
			$priority = Log::INFO;
		}

		$this->priority = $priority;
		$this->context  = $context;

		// Sanitize category if it exists.
		if (!empty($category))
		{
			$this->category = (string) strtolower(preg_replace('/[^A-Z0-9_\.-]/i', '', $category));
		}

		// Get the current call stack and back trace (without args to save memory).
		$this->callStack = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);

		// Get the date as a Date object.
		$this->date = new WbDate($date ? $date : 'now');
	}
}
