<?php

/**
 * Class Udetor_Notification.
 *
 * @since 0.8.7
 */
class Udetor_Notification
{

    /**
     * @since 0.8.7
     *
     * @var string
     */
    private static $key_options = 'udetor_hide_notifications';

    /**
     * @since 0.8.7
     *
     * @var null|array
     */
    private static $notifications_hidden = null;

    /**
     * Variable to hold the initialization state.
     *
     * @var  boolean
     */
    protected static $initialized = false;

    /**
     * Initialize functions.
     *
     * @return  void
     */
    public static function initialize()
    {
        // Do nothing if pluggable functions already initialized.
        if (self::$initialized) {
            return;
        }

        // Start Hooks
        add_action('udetor_dashboard_notifications', array(__CLASS__, 'print_notifications'));
        add_action('admin_notices', array(__CLASS__, 'print_global_notifications'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
        add_action('wp_ajax_udetor_hide_notification', array(__CLASS__, 'ajax_hide_notification'));

        // State that initialization completed.
        self::$initialized = true;
    }

    /**
     * Enqueue scripts.
     *
     * @since 0.8.7
     */
    public static function enqueue_scripts()    {

        wp_enqueue_script('udetor-notifications', Udetor_Assets::get_js_url() . 'admin/notifications.js', array('wp-util', 'jquery', 'backbone', 'underscore'), Udetor_Core::VERSION);
        self::localize_scripts();
    }

    /**
     * Localize scripts.
     *
     * @since 0.8.7
     */
    private static function localize_scripts()
    {
        wp_localize_script('udetor-notifications', 'udetor_notifications', array(
            'ajax' => admin_url('admin-ajax.php?action=udetor_hide_notification'),
        ));
    }

    /**
     * Handle ajax hide notifications.
     *
     * @since 0.8.7
     */
    public static function ajax_hide_notification()
    {
        $id = isset($_POST['id']) ? esc_html($_POST['id']) : false;
        self::hide_notification($id);

        wp_send_json_success();
    }

    /**
     * Print global notifications.
     *
     * @since 0.8.8
     */
    public static function print_global_notifications()
    {
        global $udetor_notifications;
        $notifications = $udetor_notifications;
        if(isset($notifications) && is_array($notifications)) {
            foreach ($notifications as $notification) {
                if (!$notification['global']) {
                    continue;
                }

                $id = $notification['id'];
                if (self::is_hidden($id)) {
                    continue;
                }

                $type = $notification['type'];

                $class = "udetor-$type notice notice-$type";
                if ($notification['dismissible'] && !empty($id)) {
                    $class .= ' is-dismissible';
                }

                ?>
                <div class="udetor-notice <?php echo esc_attr($class); ?>"
                     data-id="<?php echo esc_attr($notification['id']); ?>">
                    <div class="content"><?php print($notification['content']) ; ?></div>

                    <?php if ($notification['dismissible']): ?>
                        <button type="button" class="notice-dismiss"><span
                                    class="screen-reader-text"><?php esc_html_e( 'Dismiss this notice.', 'udetor' ) ?></span></button>
                    <?php endif; ?>
                </div>
                <?php
            }
        }
    }

    /**
     * Print notifications.
     *
     * @since 0.8.7
     */
    public static function print_notifications()
    {
        global $udetor_notifications;
        $notifications = $udetor_notifications;

        foreach ($notifications as $notification) {
            $id = $notification['id'];
            if (self::is_hidden($id)) {
                continue;
            }

            $type = $notification['type'];

            $class = "udetor-$type";
            if ($notification['dismissible'] && !empty($id)) {
                $class .= ' is-dismissible';
            }

            ?>
            <div class="udetor-notice <?php echo esc_attr($class); ?>"
                 data-id="<?php echo esc_attr($notification['id']); ?>">
                <div class="content"><?php print($notification['content']); ?></div>

                <?php if ($notification['dismissible']): ?>
                    <button type="button" class="notice-dismiss"><span
                                class="screen-reader-text">Dismiss this notice.</span></button>
                <?php endif; ?>
            </div>
            <?php
        }
    }

    /**
     * Hide notification.
     *
     * @since 0.8.7
     *
     * @param $id
     *
     * @return bool
     */
    private function hide_notification($id)
    {
        if (empty($id)) {
            return false;
        }

        if (self::is_hidden($id)) {
            return false;
        }

        $all = self::get_notifications_hidden();
        $all[] = $id;

        return update_option(self::$key_options, $all);
    }

    /**
     * Is notification hidden.
     *
     * @since 0.8.7
     *
     * @param $notification_id
     *
     * @return bool
     */
    private static function is_hidden($notification_id)
    {
        if (empty($notification_id)) {
            return false;
        }

        $notifications_hidden = self::get_notifications_hidden();

        foreach ($notifications_hidden as $id) {
            if ($notification_id == $id) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get all notifications hidden.
     *
     * @since 0.8.7
     *
     * @return array
     */
    private static function get_notifications_hidden()
    {
        if (self::$notifications_hidden === null) {
            self::$notifications_hidden = (array)get_option(self::$key_options);
        }

        return self::$notifications_hidden;
    }

    /**
     * Add notification.
     *
     * @since 0.8.7
     *
     * @param array $notification
     * - type: 'success', 'error', 'warning'
     * - id: unique id
     * - content: content
     * - dismissible: is dismissible
     */
    public static function add_notification($notification)
    {
        $notification = wp_parse_args($notification, array(
            'id' => '',
            'type' => 'success',
            'content' => '',
            'dismissible' => false,
            'global' => false,
        ));

        global $udetor_notifications;

        $udetor_notifications[] = $notification;
    }
}