<?php
/**
 * @version    1.0
 * @package    Udetor
 * @author     Netbase Team <support@netbaseteam.com>
 * @copyright  Copyright (C) 2014 Netbase.com. All Rights Reserved.
 * @license    GNU/GPL v2 or later http://www.gnu.org/licenses/gpl-2.0.html
 *
 */
class Udetor_Layout_Designer {
    /**
     * Variable to hold the initialization state.
     *
     * @var  boolean
     */
    protected static $initialized = false;

    /**
     * @var string
     * @since 0.9.1
     */
    public static $post_type = 'udetor-component';
    
    /**
     * Initialize functions.
     *
     * @return  void
     */
    public static function initialize() {
        // Do nothing if start functions already initialized.
        if ( self::$initialized ) {
            return;
        }

        // Register actions to do something.

        add_action( 'widgets_init', array( __CLASS__, 'register_widget' ) );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'udetor_admin_enqueue_scripts' ) );
        add_action( 'admin_init', array( __CLASS__, 'disable_revisions' ) );
        add_filter( 'get_user_option_screen_layout_udetor-component', array( __CLASS__, 'screen_layout_post' ) );
        add_action( 'admin_init', array( __CLASS__, 'enable_visual_composer' ) );
        add_action( 'admin_init', array( __CLASS__, 'handle_request_edit_content_widget' ) );

        add_filter( 'udetor_show_footer', array( __CLASS__, 'hide_footer_when_preview' ) );
        add_filter( 'udetor_show_header', array( __CLASS__, 'hide_header_when_preview' ) );
        add_action( 'udetor_wrapper_init', array( __CLASS__, 'override_template_preview' ) );
        add_action( 'wp_insert_post', array( __CLASS__, 'after_save_post_layout_builder' ), 999, 2 );
        add_action( 'wp_head', array( __CLASS__, 'add_function_insert_internal_css' ) );
        
        // State that initialization completed.
        self::$initialized = true;
    }

    /**
     * Get link panel page builder.
     *
     * @since 1.0.0
     *
     * @param $content
     * @param $builder
     * @param $hook_after_save
     * @param $extra
     *
     * @return string
     */
    public static function get_link_panel_page_builder( $content, $builder = 'default', $hook_after_save = array(), $extra = null ) {
        $post_ID = self::get_layout_builder_available();

        /**
         * Setup default.
         */
        $meta = array(
            'udetor_hook_after_save_post' => $hook_after_save,
            'udetor_extra'                => $extra
        );

        $args = array(
            'ID' => $post_ID,
        );

        if ( $builder != 'so' ) {
            $args['post_content'] = $content;
        } else {
            $meta['panels_data'] = $content;
        }

        if ( $builder != 'vc' ) {
            $meta['_wpb_vc_js_status'] = 'false';
        } else {
            $meta['_wpb_vc_js_status'] = 'true';
        }

        $args['meta_input'] = $meta;

        wp_update_post( $args );
        if ( is_wp_error( $post_ID ) ) {
            wp_die( esc_html__( 'Something went wrong!', 'udetor' ) );
        }

        return admin_url( "post.php?post=$post_ID&action=edit" );
    }

    /**
     * Get content layout builder.
     *
     * @since 1.0.0
     *
     * @param $post_ID
     *
     * @return string|mixed
     */
    public static function get_content( $post_ID ) {
        $p = get_post( $post_ID );

        if ( ! $p ) {
            return '';
        }

        $builder = self::detect_page_builder( $post_ID );

        if ( $builder != 'so' ) {
            return $p->post_content;
        }

        return get_post_meta( $post_ID, 'panels_data', true );
    }

    /**
     * Render content to HTML.
     *
     * @since 1.0.0
     *
     * @param $content
     * @param $builder
     * @param $fake_post_id
     *
     * @return string
     */
    public static function render_content( $content, $builder, $fake_post_id ) {
        if ( $builder == 'so' ) {
            if ( ! function_exists( 'siteorigin_panels_render' ) ) {
                return "<pre class='notification-error'>"
                    . sprintf( esc_html__( 'Please install <a href="%s" target="_blank">SiteOrigin Page Builder</a>', 'udetor' ), 'https://siteorigin.com/page-builder/' )
                    . "</pre>";
            }

            return siteorigin_panels_render( $fake_post_id, true, $content );
        }

        $rendered = do_shortcode( $content );

        if ( $builder == 'vc' && function_exists( 'visual_composer' ) ) {
            $custom_css = visual_composer()->parseShortcodesCustomCss( $content );

            if ( ! empty( $custom_css ) ) {
                $script = '<script type="text/javascript">
                udetor_insert_internal_css("' . $custom_css . '");
			</script>';

                $rendered .= $script;
            }
        }

        return $rendered;
    }

    /**
     * Detect page builder.
     *
     * @since 1.0.0
     *
     * @param $post_ID
     *
     * @return string so: Site Origin, vc: Visual Composer, default: HTML
     */
    public static function detect_page_builder( $post_ID ) {
        $vc = get_post_meta( $post_ID, '_wpb_vc_js_status', true );
        if ( ! empty( $vc ) && $vc == 'true' ) {
            return 'vc';
        }

        $site_origin = get_post_meta( $post_ID, 'panels_data', true );
        if ( ! empty( $site_origin ) ) {
            return 'so';
        }

        return 'default';
    }

    /**
     * Get post layout builder.
     *
     * @since 0.9.0
     *
     * @return WP_Error|int
     */
    public static function get_layout_builder_available() {
        $max_post_count = 3;
        $query          = new WP_Query( array(
            'posts_per_page' => - 1,
            'post_type'      => self::$post_type,
            'order'          => 'ASC',
            'orderby'        => 'modified'
        ) );

        $postID = false;

        $posts = $query->get_posts();
        if ( ! $posts || count( $posts ) < $max_post_count ) {
            $args = array(
                'post_title'   => time(),
                'post_content' => '',
                'post_status'  => 'publish',
                'post_type'    => self::$post_type,
                'meta_input'   => array(
                    '_wpb_vc_js_status'       => 'true',//Enable default visual composer
                    'udetor_hook_after_save_post' => '',
                )
            );

            $post   = wp_insert_post( $args, new WP_Error( '5000', esc_html__( 'Can not create new post', 'udetor' ) ) );
            $postID = $post;
        }

        $post = $posts[0];
        if ( $post instanceof WP_Post ) {
            $postID = $post->ID;
        }

        if ( ! $postID ) {
            return new WP_Error( '5000', esc_html__( 'Layout designer not available now! Please try again later :)', 'udetor' ) );
        }

        update_post_meta( $postID, 'udetor_hook_after_save_post', '' );

        return $postID;
    }

    /**
     * Get link page layout builder.
     *
     * @since 0.9.0
     *
     * @param $content
     * @param $hook_after_save
     * @param $extra
     *
     * @return string|WP_Error
     */
    public static function get_link_page_layout_builder( $content, $hook_after_save = array(), $extra = null ) {
        $post_id = self::get_layout_builder_available();

        if ( is_wp_error( $post_id ) ) {
            return $post_id;
        }

        $meta = array(
            'udetor_hook_after_save_post' => $hook_after_save,
        );

        if ( $extra !== null ) {
            $meta['udetor_extra'] = $extra;
        }

        wp_update_post( array(
            'ID'           => $post_id,
            'post_content' => $content,
            'meta_input'   => $meta,
        ) );

        return admin_url( "post.php?post=$post_id&action=edit" );
    }

    /**
     * Add script function insert internal css.
     *
     * @since 1.0.1
     */
    public static function add_function_insert_internal_css() {
        ?>
        <script type="text/javascript">
            function udetor_insert_internal_css(css) {
                var udetor_style = document.createElement("style");
                udetor_style.type = "text/css";
                udetor_style.setAttribute('data-type', 'udetor-internal-css');
                var udetor_style_content = document.createTextNode(css);
                udetor_style.appendChild(udetor_style_content);
                document.head.appendChild(udetor_style);
            }
        </script>
        <?php
    }

    /**
     * Handler after save post layout builder (call hook).
     *
     * @since 0.9.0
     *
     * @param $post_ID
     * @param $post
     */
    public static function after_save_post_layout_builder( $post_ID, $post ) {
        if ( $post->post_type != self::$post_type ) {
            return;
        }

        $hook_after_save_post = get_post_meta( $post_ID, 'udetor_hook_after_save_post', true );

        if ( empty( $hook_after_save_post ) ) {
            return;
        }

        if ( is_callable( $hook_after_save_post ) ) {
            $extra = get_post_meta( $post_ID, 'udetor_extra', true );

            call_user_func( $hook_after_save_post, array( $post_ID, $extra ) );
        }
    }

    /**
     * Enable visual composer.
     *
     * @since 0.8.2
     */
    public static function enable_visual_composer() {
        $post_id = isset( $_REQUEST['post'] ) ? $_REQUEST['post'] : false;
        if ( get_post_type( $post_id ) !== self::$post_type ) {
            return;
        }

        add_filter( 'vc_role_access_with_post_types_get_state', '__return_true' );
        add_filter( 'vc_role_access_with_backend_editor_get_state', '__return_true' );
        add_filter( 'vc_role_access_with_frontend_editor_get_state', '__return_true' );
        add_filter( 'vc_check_post_type_validation', '__return_true' );
    }

    /**
     * Override template preview.
     *
     * @since 0.9.0
     */
    public static function override_template_preview() {
        $post_type = get_post_type();
        if ( $post_type == self::$post_type ) {
            Udetor_Template_Helper::template( 'layout-builder/preview.php', array(), true );
            exit();
        }
    }

    /**
     * Hide footer when preview.
     *
     * @since 0.9.0
     *
     * @param $show
     *
     * @return bool
     */
    public static function hide_footer_when_preview( $show ) {
        if ( get_post_type() !== self::$post_type ) {
            return $show;
        }

        return false;
    }

    /**
     * Hide header when preview.
     *
     * @since 0.9.0
     *
     * @param $show
     *
     * @return bool
     */
    public static function hide_header_when_preview( $show ) {
        if ( get_post_type() !== self::$post_type ) {
            return $show;
        }

        return false;
    }

    /**
     * Action delete page link to widget.
     *
     * @since 0.8.2
     *
     * @param $widget_id
     * @param $sidebar_id
     * @param $id_base
     */
    public static function delete_page_linking_widget( $widget_id, $sidebar_id, $id_base ) {
        if ( $id_base !== Udetor_Widget_Designer::get_id_base() ) {
            return;
        }

        $id = str_replace( $id_base . '-', '', $widget_id );
        if ( ! is_numeric( $id ) ) {
            return;
        }

        Udetor_WidgetDesigner_Linking_Layout::delete_page_linking_widget( $id );
    }

    /**
     * Handle request go to page builder.
     *
     * @since 0.8.2
     */
    public static function handle_request_edit_content_widget() {
        Udetor_WidgetDesigner_Linking_Layout::handle_request_edit_content_widget();
    }

    /**
     * Add filter screen layout edit post type udetor-component is 1 column.
     *
     * @since 0.8.2
     *
     * @return int
     */
    public static function screen_layout_post() {
        return 1;
    }

    /**
     * Enqueue scripts.
     *
     * @since 0.8.2
     */
    public static function udetor_admin_enqueue_scripts() {
        $screen = get_current_screen();

        if ( $screen->post_type !== self::$post_type ) {
            return;
        }

        wp_enqueue_style( 'udetor-edit-component', Udetor_Assets::get_css_url() . 'admin/udetor-layout-builder.css' );
        do_action( 'udetor_enqueue_scripts_panel_layout_builder' );
    }

    /**
     * Disable revisions.
     *
     * @since 0.8.2
     */
    public static function disable_revisions() {
        remove_post_type_support( self::$post_type, 'revisions' );
    }

    

    /**
     * Register widget.
     *
     * @since 0.8.2
     */
    public static function register_widget() {
        register_widget( 'Udetor_WidgetDesigner' );
    }
    
}