<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package  Udetor
 * @since    1.0
 */
class Udetor_Helper {
    /**
     * Variable to hold the initialization state.
     *
     * @var  boolean
     */
    protected static $initialized = false;
    
    /**
     * Initialize functions.
     *
     * @return  void
     */
    public static function initialize() {
        // Do nothing if start functions already initialized.
        if ( self::$initialized ) {
            return;
        }

        // Register actions to do something.        
        add_action( 'body_class', array( __CLASS__, 'udetor_body_classes' ) );
        add_action( 'wp_head', array( __CLASS__, 'udetor_pingback_header' ) );        

        add_filter( 'FHEE__content_espresso_events_details_template__the_content',  array( __CLASS__, 'udetor_events_details_the_content' ));
        add_filter( 'the_content_more_link', array( __CLASS__, 'udetor_modify_read_more_link' ) );   

        add_filter( 'pre_get_posts', array( __CLASS__, 'udetor_query_cpt_search' ) );     
        
        // State that initialization completed.
        self::$initialized = true;
    }  

    /**
     * This function modifies the main WordPress query to include an array of 
     * post types instead of the default 'post' post type.
     *
     * @param object $query  The original query.
     * @return object $query The amended query.
     */
    public static function udetor_query_cpt_search( $query ) {
        
        if ( $query->is_search ) {
        $query->set( 'post_type', array( 'post', 'sfwd-courses', 'products', 'sfwd-lessons' ) );
        }
        
        return $query;
        
    }
      

    public static function udetor_modify_read_more_link() {
        return '<a class="udetor-more-link" href="' . get_permalink() . '">'. esc_html__('Read More', 'udetor' ) .'</a>';
    }
    
    /**
     * Adds custom classes to the array of body classes.
     *
     * @param array $classes Classes for the body element.
     * @return array
     */
    public static function udetor_body_classes( $classes ) {
        // Adds a class of group-blog to blogs with more than 1 published author.
        $udetor_site_layout     = get_theme_mod( 'udetor_site_layout');
        $offset = get_theme_mod('udetor_layout_offset_width');

        if ( is_multi_author() ) {
            $classes[] = 'group-blog';
        }
        
        // Adds a class of hfeed to non-singular pages.
        if ( ! is_singular() ) {
            $classes[] = 'hfeed';
        }
        
        if($udetor_site_layout=='content-boxed'){
             $classes[] = 'content-boxed';
        }
        if($offset){
            $classes[] = 'udetor-offset';
        }
        
        return $classes;
    }

    /**
     * Add a pingback url auto-discovery header for singularly identifiable articles.
     */
    public static function udetor_pingback_header() {
        if ( is_singular() && pings_open() ) {
            echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
        }
    }

    public static function udetor_get_all_post_type(){
        $args = array('public'   => true,
                        /*'_builtin' => false*/
                    );
                    $post_types = get_post_types( $args,'objects');
                    $all_posttypes = array();
                    foreach($post_types as $post_type_obj)
                    {
                        $labels = get_post_type_labels( $post_type_obj );
                        $all_posttypes[$post_type_obj->name]=$labels->name; 
                        
                    }
                    return $all_posttypes;
    }

    /**
     * Get all registered sidebars.
     *
     * @return  array
     */
    public static function get_sidebars() {
        global $wp_registered_sidebars;

        // Get custom sidebars.
        $custom_sidebars = get_option( 'udetor_theme_sidebars' );

        // Prepare output.
        $output = array();

        if ( is_customize_preview() ) {
            $output[] = esc_html__( '-- Select Sidebar --', 'udetor' );
        }

        if ( ! empty( $wp_registered_sidebars ) ) {
            foreach ( $wp_registered_sidebars as $sidebar ) {
                $output[ $sidebar['id'] ] = $sidebar['name'];
            }
        }

        if ( ! empty( $custom_sidebars ) ) {
            foreach ( $custom_sidebars as $sidebar ) {
                $output[ $sidebar['id'] ] = $sidebar['name'];
            }
        }

        return $output;
    }

    public static function udetor_get_layout_class ( ) {
        global $post;
        global $woocommerce;
        $classes               = '';
        $page_layout_admin     = get_theme_mod( 'udetor_standard_page_sidebar_layout');        
        $blog_single_layout     = get_theme_mod( 'udetor_blog_single_content_layout','right-sidebar');
        $blog_list_layout     = get_theme_mod( 'udetor_blog_list_layout');
        $udetor_courses_sidebar_layout     = get_theme_mod( 'udetor_courses_sidebar_layout');
        $udetor_courses_detail_sidebar_layout = get_theme_mod('udetor_courses_detail_sidebar_layout');
        $udetor_courses_lesson_layouts = get_theme_mod('udetor_courses_lesson_layouts','right-sidebar' );
        $udetor_woo_archive_layout = get_theme_mod('udetor_woo_archive_layout','left-sidebar');  
        $udetor_woo_detail_layout = get_theme_mod('udetor_woo_detail_layout','no-sidebar');     

        // Pages
        if ( is_page() ){
            $classes = $page_layout_admin;
            $page_meta =  esc_html( get_post_meta(  $post->ID, 'sidebar_layout', true) );

            if ( $page_meta == 'default' || $page_meta == '' && $page_layout_admin !='' ) {
                $classes = $page_layout_admin;
            }else {
                $classes = $page_meta;
            }
            if ( is_plugin_active( 'buddypress/bp-loader.php' ) ) {
                if(bp_current_component('register')){
                    $classes= 'no-sidebar';
                }
            }
        }

        // Single Post
        if ( is_single() ) {
            if ( $blog_single_layout ) {
                $classes = $blog_single_layout;
            } else {
                $classes = 'right-sidebar';
            }
        }

        // Archive
        if ( (is_archive() || is_author()) & !is_front_page() ) {
            if ( $blog_list_layout !== '' ){
                $classes = $blog_list_layout;
            } 
            
        }
        if((is_archive()  && !is_author()) && !is_front_page()){
            if (get_post_type() == 'sfwd-courses' || get_post_type() == 'sfwd-lessons' || get_post_type() == 'sfwd-topic' || get_post_type() == 'sfwd-quiz') {
                if($udetor_courses_sidebar_layout !=''){
                    $classes = $udetor_courses_sidebar_layout;
                }

            }

        }
        if((is_single()  && !is_author()) && !is_front_page()){
            if (get_post_type() == 'sfwd-courses' ) {
                if($udetor_courses_detail_sidebar_layout !=''){
                        $classes = $udetor_courses_detail_sidebar_layout; 
                }
            }

        }

        if((is_single()  && !is_author()) && !is_front_page()){
            if (get_post_type() == 'sfwd-lessons' || get_post_type() == 'sfwd-topic' || get_post_type() == 'sfwd-quiz') {
                if($udetor_courses_lesson_layouts !=''){
                        $classes = $udetor_courses_lesson_layouts; 
                }
            }
        }

        // WooCommerce
        if ( $woocommerce ) {
            if ( $woocommerce && is_shop() || $woocommerce && is_product_category() || $woocommerce && is_product_tag() ) {
                if ( $udetor_woo_archive_layout ){
                    $classes = $udetor_woo_archive_layout;
                } 
                
            }
            if ( $woocommerce && is_product() ) {
                if ( $udetor_woo_detail_layout ) {
                    $classes = $udetor_woo_detail_layout;
                }
            }
        }       

        return $classes;
    }

    /**
     * Prints HTML with meta information for the current post-date/time and author.
     */
    public static function udetor_meta_posted_on() {
        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
        $time_string = sprintf( $time_string,
            esc_attr( get_the_date( 'c' ) ),
            esc_html( get_the_date() ),
            esc_attr( get_the_modified_date( 'c' ) ),
            esc_html( get_the_modified_date() )
        );
        $posted_on = sprintf(
            _x( '%s', 'post date', 'udetor' ), $time_string 
        );
        
        ?>
        <div class="blog-single-meta">            
            <?php 
            if(get_the_author()){
                    echo '<div class="meta-item">';

                    $job_profile = get_the_author_meta( 'job_profile' );
                    $getjobauthor='';
                    if ( $job_profile && $job_profile != '' ) {
                        $getjobauthor = sprintf(
                                _x( '%s', 'post author', 'udetor' ),
                                '<span class="author-job">' . esc_html( $job_profile ) . '</span>'
                            );
                    }
                    $getauthor = sprintf(
                        _x( '%s', 'post author', 'udetor' ),
                        '<span class="author-name">' . esc_html( get_the_author() ) . '</span>'
                    );
                    echo '<div class="meta-label">' . esc_html__( 'Instructor', 'udetor' ) . '</div>';
                    echo '<div class="meta-author meta-content">'.get_avatar( get_the_author_meta('email'), '51' ).'<div class="author-bio">'. $getauthor . ''. $getjobauthor .'</div></div>';
                echo '</div>';
            }
              
            $categories_list = get_the_category_list( esc_html__( ', ', 'udetor' ) );
            if ( $categories_list ) {
                echo '<div class="meta-item">';
                echo '<div class="meta-label">' . esc_html__( 'Category', 'udetor' ) . '</div>';
                printf( '<div class="meta-categories-links meta-content">' . esc_html__( '%1$s', 'udetor' ) . '</div>', $categories_list );
                echo '</div>';
            }                     
            
            echo '<div class="meta-item">';
                echo '<div class="meta-label">' . esc_html__( 'Date', 'udetor' ) . '</div>'; 
                echo '<div class="meta-content">'. $posted_on .'</div>';
            echo '</div>';
             
            if(function_exists('the_views')) { 
                echo '<div class="meta-item">';
                    echo '<div class="meta-label">' . esc_html__( 'Views', 'udetor' ) . '</div>';
                    echo '<div class="meta-content">';
                        the_views(); 
                    echo '</div>';
                echo '</div>';
            }
            ?>
        </div>
        <?php
    }   
    
    /**
     * Parse url youtube to id.
     *
     * @since 1.0.0
     *
     * @param $url
     *
     * @return mixed
     */

    public static function udetor_parse_id_youtube($url){
        if (preg_match('%(?:youtube(?:-nocookie)?\.com/(?:[^/]+/.+/|(?:v|e(?:mbed)?)/|.*[?&]v=)|youtu\.be/)([^"&?/ ]{11})%i', $url, $match)) {
            $video_id = $match[1];

            return $video_id;
        }

        return false;
    }


    /**
     * Redirect to url.
     *
     * @since 0.8.9
     *
     * @param $url
     */

    public static function udetor_core_redirect($url)
    {
        if (headers_sent()) {
            echo "<meta http-equiv='refresh' content='0;URL=$url' />";
        } else {
            wp_redirect($url);
        }

        exit();
    }    
    
    /** override event */
    public static function udetor_events_details_the_content(){    
        the_content();

    }    

    /*
     * Get list of Google fonts.
     *
     * @return  array
     */
    public static function google_fonts() {
        return array(
            'ABeeZee' => array( 400 ),
            'Abel' => array( 400 ),
            'Abril Fatface' => array( 400 ),
            'Aclonica' => array( 400 ),
            'Acme' => array( 400 ),
            'Actor' => array( 400 ),
            'Adamina' => array( 400 ),
            'Advent Pro' => array( 100, 200, 300, 400, 500, 600, 700 ),
            'Aguafina Script' => array( 400 ),
            'Akronim' => array( 400 ),
            'Aladin' => array( 400 ),
            'Aldrich' => array( 400 ),
            'Alef' => array( 400, 700 ),
            'Alegreya' => array( 400, 700, 900 ),
            'Alegreya SC' => array( 400, 700, 900 ),
            'Alegreya Sans' => array( 100, 300, 400, 500, 700, 800, 900 ),
            'Alegreya Sans SC' => array( 100, 300, 400, 500, 700, 800, 900 ),
            'Alex Brush' => array( 400 ),
            'Alfa Slab One' => array( 400 ),
            'Alice' => array( 400 ),
            'Alike' => array( 400 ),
            'Alike Angular' => array( 400 ),
            'Allan' => array( 400, 700 ),
            'Allerta' => array( 400 ),
            'Allerta Stencil' => array( 400 ),
            'Allura' => array( 400 ),
            'Almendra' => array( 400, 700 ),
            'Almendra Display' => array( 400 ),
            'Almendra SC' => array( 400 ),
            'Amarante' => array( 400 ),
            'Amaranth' => array( 400, 700 ),
            'Amatic SC' => array( 400, 700 ),
            'Amethysta' => array( 400 ),
            'Amiri' => array( 400, 700 ),
            'Amita' => array( 400, 700 ),
            'Anaheim' => array( 400 ),
            'Andada' => array( 400 ),
            'Andika' => array( 400 ),
            'Angkor' => array( 400 ),
            'Annie Use Your Telescope' => array( 400 ),
            'Anonymous Pro' => array( 400, 700 ),
            'Antic' => array( 400 ),
            'Antic Didone' => array( 400 ),
            'Antic Slab' => array( 400 ),
            'Anton' => array( 400 ),
            'Arapey' => array( 400 ),
            'Arbutus' => array( 400 ),
            'Arbutus Slab' => array( 400 ),
            'Architects Daughter' => array( 400 ),
            'Archivo Black' => array( 400 ),
            'Archivo Narrow' => array( 400, 700 ),
            'Arimo' => array( 400, 700 ),
            'Arizonia' => array( 400 ),
            'Armata' => array( 400 ),
            'Artifika' => array( 400 ),
            'Arvo' => array( 400, 700 ),
            'Arya' => array( 400, 700 ),
            'Asap' => array( 400, 700 ),
            'Asar' => array( 400 ),
            'Asset' => array( 400 ),
            'Astloch' => array( 400, 700 ),
            'Asul' => array( 400, 700 ),
            'Atomic Age' => array( 400 ),
            'Aubrey' => array( 400 ),
            'Audiowide' => array( 400 ),
            'Autour One' => array( 400 ),
            'Average' => array( 400 ),
            'Average Sans' => array( 400 ),
            'Averia Gruesa Libre' => array( 400 ),
            'Averia Libre' => array( 300, 400, 700 ),
            'Averia Sans Libre' => array( 300, 400, 700 ),
            'Averia Serif Libre' => array( 300, 400, 700 ),
            'Bad Script' => array( 400 ),
            'Balthazar' => array( 400 ),
            'Bangers' => array( 400 ),
            'Basic' => array( 400 ),
            'Battambang' => array( 400, 700 ),
            'Baumans' => array( 400 ),
            'Bayon' => array( 400 ),
            'Belgrano' => array( 400 ),
            'Belleza' => array( 400 ),
            'BenchNine' => array( 300, 400, 700 ),
            'Bentham' => array( 400 ),
            'Berkshire Swash' => array( 400 ),
            'Bevan' => array( 400 ),
            'Bigelow Rules' => array( 400 ),
            'Bigshot One' => array( 400 ),
            'Bilbo' => array( 400 ),
            'Bilbo Swash Caps' => array( 400 ),
            'Biryani' => array( 200, 300, 400, 600, 700, 800, 900 ),
            'Bitter' => array( 400, 700 ),
            'Black Ops One' => array( 400 ),
            'Bokor' => array( 400 ),
            'Bonbon' => array( 400 ),
            'Boogaloo' => array( 400 ),
            'Bowlby One' => array( 400 ),
            'Bowlby One SC' => array( 400 ),
            'Brawler' => array( 400 ),
            'Bree Serif' => array( 400 ),
            'Bubblegum Sans' => array( 400 ),
            'Bubbler One' => array( 400 ),
            'Buda' => array( 300 ),
            'Buenard' => array( 400, 700 ),
            'Butcherman' => array( 400 ),
            'Butterfly Kids' => array( 400 ),
            'Cabin' => array( 400, 500, 600, 700 ),
            'Cabin Condensed' => array( 400, 500, 600, 700 ),
            'Cabin Sketch' => array( 400, 700 ),
            'Caesar Dressing' => array( 400 ),
            'Cagliostro' => array( 400 ),
            'Calligraffitti' => array( 400 ),
            'Cambay' => array( 400, 700 ),
            'Cambo' => array( 400 ),
            'Candal' => array( 400 ),
            'Cantarell' => array( 400, 700 ),
            'Cantata One' => array( 400 ),
            'Cantora One' => array( 400 ),
            'Capriola' => array( 400 ),
            'Cardo' => array( 400, 700 ),
            'Carme' => array( 400 ),
            'Carrois Gothic' => array( 400 ),
            'Carrois Gothic SC' => array( 400 ),
            'Carter One' => array( 400 ),
            'Catamaran' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Caudex' => array( 400, 700 ),
            'Caveat' => array( 400, 700 ),
            'Caveat Brush' => array( 400 ),
            'Cedarville Cursive' => array( 400 ),
            'Ceviche One' => array( 400 ),
            'Changa One' => array( 400 ),
            'Chango' => array( 400 ),
            'Chau Philomene One' => array( 400 ),
            'Chela One' => array( 400 ),
            'Chelsea Market' => array( 400 ),
            'Chenla' => array( 400 ),
            'Cherry Cream Soda' => array( 400 ),
            'Cherry Swash' => array( 400, 700 ),
            'Chewy' => array( 400 ),
            'Chicle' => array( 400 ),
            'Chivo' => array( 400, 900 ),
            'Chonburi' => array( 400 ),
            'Cinzel' => array( 400, 700, 900 ),
            'Cinzel Decorative' => array( 400, 700, 900 ),
            'Clicker Script' => array( 400 ),
            'Coda' => array( 400, 800 ),
            'Coda Caption' => array( 800 ),
            'Codystar' => array( 300, 400 ),
            'Combo' => array( 400 ),
            'Comfortaa' => array( 300, 400, 700 ),
            'Coming Soon' => array( 400 ),
            'Concert One' => array( 400 ),
            'Condiment' => array( 400 ),
            'Content' => array( 400, 700 ),
            'Contrail One' => array( 400 ),
            'Convergence' => array( 400 ),
            'Cookie' => array( 400 ),
            'Copse' => array( 400 ),
            'Corben' => array( 400, 700 ),
            'Courgette' => array( 400 ),
            'Cousine' => array( 400, 700 ),
            'Coustard' => array( 400, 900 ),
            'Covered By Your Grace' => array( 400 ),
            'Crafty Girls' => array( 400 ),
            'Creepster' => array( 400 ),
            'Crete Round' => array( 400 ),
            'Crimson Text' => array( 400, 600, 700 ),
            'Croissant One' => array( 400 ),
            'Crushed' => array( 400 ),
            'Cuprum' => array( 400, 700 ),
            'Cutive' => array( 400 ),
            'Cutive Mono' => array( 400 ),
            'Damion' => array( 400 ),
            'Dancing Script' => array( 400, 700 ),
            'Dangrek' => array( 400 ),
            'Dawning of a New Day' => array( 400 ),
            'Days One' => array( 400 ),
            'Dekko' => array( 400 ),
            'Delius' => array( 400 ),
            'Delius Swash Caps' => array( 400 ),
            'Delius Unicase' => array( 400, 700 ),
            'Della Respira' => array( 400 ),
            'Denk One' => array( 400 ),
            'Devonshire' => array( 400 ),
            'Dhurjati' => array( 400 ),
            'Didact Gothic' => array( 400 ),
            'Diplomata' => array( 400 ),
            'Diplomata SC' => array( 400 ),
            'Domine' => array( 400, 700 ),
            'Donegal One' => array( 400 ),
            'Doppio One' => array( 400 ),
            'Dorsa' => array( 400 ),
            'Dosis' => array( 200, 300, 400, 500, 600, 700, 800 ),
            'Dr Sugiyama' => array( 400 ),
            'Droid Sans' => array( 400, 700 ),
            'Droid Sans Mono' => array( 400 ),
            'Droid Serif' => array( 400, 700 ),
            'Duru Sans' => array( 400 ),
            'Dynalight' => array( 400 ),
            'EB Garamond' => array( 400 ),
            'Eagle Lake' => array( 400 ),
            'Eater' => array( 400 ),
            'Economica' => array( 400, 700 ),
            'Eczar' => array( 400, 500, 600, 700, 800 ),
            'Ek Mukta' => array( 200, 300, 400, 500, 600, 700, 800 ),
            'Electrolize' => array( 400 ),
            'Elsie' => array( 400, 900 ),
            'Elsie Swash Caps' => array( 400, 900 ),
            'Emblema One' => array( 400 ),
            'Emilys Candy' => array( 400 ),
            'Engagement' => array( 400 ),
            'Englebert' => array( 400 ),
            'Enriqueta' => array( 400, 700 ),
            'Erica One' => array( 400 ),
            'Esteban' => array( 400 ),
            'Euphoria Script' => array( 400 ),
            'Ewert' => array( 400 ),
            'Exo' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Exo 2' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Expletus Sans' => array( 400, 500, 600, 700 ),
            'Fanwood Text' => array( 400 ),
            'Fascinate' => array( 400 ),
            'Fascinate Inline' => array( 400 ),
            'Faster One' => array( 400 ),
            'Fasthand' => array( 400 ),
            'Fauna One' => array( 400 ),
            'Federant' => array( 400 ),
            'Federo' => array( 400 ),
            'Felipa' => array( 400 ),
            'Fenix' => array( 400 ),
            'Finger Paint' => array( 400 ),
            'Fira Mono' => array( 400, 700 ),
            'Fira Sans' => array( 300, 400, 500, 700 ),
            'Fjalla One' => array( 400 ),
            'Fjord One' => array( 400 ),
            'Flamenco' => array( 300, 400 ),
            'Flavors' => array( 400 ),
            'Fondamento' => array( 400 ),
            'Fontdiner Swanky' => array( 400 ),
            'Forum' => array( 400 ),
            'Francois One' => array( 400 ),
            'Freckle Face' => array( 400 ),
            'Fredericka the Great' => array( 400 ),
            'Fredoka One' => array( 400 ),
            'Freehand' => array( 400 ),
            'Fresca' => array( 400 ),
            'Frijole' => array( 400 ),
            'Fruktur' => array( 400 ),
            'Fugaz One' => array( 400 ),
            'GFS Didot' => array( 400 ),
            'GFS Neohellenic' => array( 400, 700 ),
            'Gabriela' => array( 400 ),
            'Gafata' => array( 400 ),
            'Galdeano' => array( 400 ),
            'Galindo' => array( 400 ),
            'Gentium Basic' => array( 400, 700 ),
            'Gentium Book Basic' => array( 400, 700 ),
            'Geo' => array( 400 ),
            'Geostar' => array( 400 ),
            'Geostar Fill' => array( 400 ),
            'Germania One' => array( 400 ),
            'Gidugu' => array( 400 ),
            'Gilda Display' => array( 400 ),
            'Give You Glory' => array( 400 ),
            'Glass Antiqua' => array( 400 ),
            'Glegoo' => array( 400, 700 ),
            'Gloria Hallelujah' => array( 400 ),
            'Goblin One' => array( 400 ),
            'Gochi Hand' => array( 400 ),
            'Gorditas' => array( 400, 700 ),
            'Goudy Bookletter 1911' => array( 400 ),
            'Graduate' => array( 400 ),
            'Grand Hotel' => array( 400 ),
            'Gravitas One' => array( 400 ),
            'Great Vibes' => array( 400 ),
            'Griffy' => array( 400 ),
            'Gruppo' => array( 400 ),
            'Gudea' => array( 400, 700 ),
            'Gurajada' => array( 400 ),
            'Habibi' => array( 400 ),
            'Halant' => array( 300, 400, 500, 600, 700 ),
            'Hammersmith One' => array( 400 ),
            'Hanalei' => array( 400 ),
            'Hanalei Fill' => array( 400 ),
            'Handlee' => array( 400 ),
            'Hanuman' => array( 400, 700 ),
            'Happy Monkey' => array( 400 ),
            'Headland One' => array( 400 ),
            'Henny Penny' => array( 400 ),
            'Herr Von Muellerhoff' => array( 400 ),
            'Hind' => array( 300, 400, 500, 600, 700 ),
            'Hind Siliguri' => array( 300, 400, 500, 600, 700 ),
            'Hind Vadodara' => array( 300, 400, 500, 600, 700 ),
            'Holtwood One SC' => array( 400 ),
            'Homemade Apple' => array( 400 ),
            'Homenaje' => array( 400 ),
            'IM Fell DW Pica' => array( 400 ),
            'IM Fell DW Pica SC' => array( 400 ),
            'IM Fell Double Pica' => array( 400 ),
            'IM Fell Double Pica SC' => array( 400 ),
            'IM Fell English' => array( 400 ),
            'IM Fell English SC' => array( 400 ),
            'IM Fell French Canon' => array( 400 ),
            'IM Fell French Canon SC' => array( 400 ),
            'IM Fell Great Primer' => array( 400 ),
            'IM Fell Great Primer SC' => array( 400 ),
            'Iceberg' => array( 400 ),
            'Iceland' => array( 400 ),
            'Imprima' => array( 400 ),
            'Inconsolata' => array( 400, 700 ),
            'Inder' => array( 400 ),
            'Indie Flower' => array( 400 ),
            'Inika' => array( 400, 700 ),
            'Inknut Antiqua' => array( 300, 400, 500, 600, 700, 800, 900 ),
            'Irish Grover' => array( 400 ),
            'Istok Web' => array( 400, 700 ),
            'Italiana' => array( 400 ),
            'Italianno' => array( 400 ),
            'Itim' => array( 400 ),
            'Jacques Francois' => array( 400 ),
            'Jacques Francois Shadow' => array( 400 ),
            'Jaldi' => array( 400, 700 ),
            'Jim Nightshade' => array( 400 ),
            'Jockey One' => array( 400 ),
            'Jolly Lodger' => array( 400 ),
            'Josefin Sans' => array( 100, 300, 400, 600, 700 ),
            'Josefin Slab' => array( 100, 300, 400, 600, 700 ),
            'Joti One' => array( 400 ),
            'Judson' => array( 400, 700 ),
            'Julee' => array( 400 ),
            'Julius Sans One' => array( 400 ),
            'Junge' => array( 400 ),
            'Jura' => array( 300, 400, 500, 600 ),
            'Just Another Hand' => array( 400 ),
            'Just Me Again Down Here' => array( 400 ),
            'Kadwa' => array( 400, 700 ),
            'Kalam' => array( 300, 400, 700 ),
            'Kameron' => array( 400, 700 ),
            'Kanit' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Kantumruy' => array( 300, 400, 700 ),
            'Karla' => array( 400, 700 ),
            'Karma' => array( 300, 400, 500, 600, 700 ),
            'Kaushan Script' => array( 400 ),
            'Kavoon' => array( 400 ),
            'Kdam Thmor' => array( 400 ),
            'Keania One' => array( 400 ),
            'Kelly Slab' => array( 400 ),
            'Kenia' => array( 400 ),
            'Khand' => array( 300, 400, 500, 600, 700 ),
            'Khmer' => array( 400 ),
            'Khula' => array( 300, 400, 600, 700, 800 ),
            'Kite One' => array( 400 ),
            'Knewave' => array( 400 ),
            'Kotta One' => array( 400 ),
            'Koulen' => array( 400 ),
            'Kranky' => array( 400 ),
            'Kreon' => array( 300, 400, 700 ),
            'Kristi' => array( 400 ),
            'Krona One' => array( 400 ),
            'Kurale' => array( 400 ),
            'La Belle Aurore' => array( 400 ),
            'Laila' => array( 300, 400, 500, 600, 700 ),
            'Lakki Reddy' => array( 400 ),
            'Lancelot' => array( 400 ),
            'Lateef' => array( 400 ),
            'Lato' => array( 100, 300, 400, 700, 900 ),
            'League Script' => array( 400 ),
            'Leckerli One' => array( 400 ),
            'Ledger' => array( 400 ),
            'Lekton' => array( 400, 700 ),
            'Lemon' => array( 400 ),
            'Libre Baskerville' => array( 400, 700 ),
            'Life Savers' => array( 400, 700 ),
            'Lilita One' => array( 400 ),
            'Lily Script One' => array( 400 ),
            'Limelight' => array( 400 ),
            'Linden Hill' => array( 400 ),
            'Lobster' => array( 400 ),
            'Lobster Two' => array( 400, 700 ),
            'Londrina Outline' => array( 400 ),
            'Londrina Shadow' => array( 400 ),
            'Londrina Sketch' => array( 400 ),
            'Londrina Solid' => array( 400 ),
            'Lora' => array( 400, 700 ),
            'Love Ya Like A Sister' => array( 400 ),
            'Loved by the King' => array( 400 ),
            'Lovers Quarrel' => array( 400 ),
            'Luckiest Guy' => array( 400 ),
            'Lusitana' => array( 400, 700 ),
            'Lustria' => array( 400 ),
            'Macondo' => array( 400 ),
            'Macondo Swash Caps' => array( 400 ),
            'Magra' => array( 400, 700 ),
            'Maiden Orange' => array( 400 ),
            'Mako' => array( 400 ),
            'Mallanna' => array( 400 ),
            'Mandali' => array( 400 ),
            'Marcellus' => array( 400 ),
            'Marcellus SC' => array( 400 ),
            'Marck Script' => array( 400 ),
            'Margarine' => array( 400 ),
            'Marko One' => array( 400 ),
            'Marmelad' => array( 400 ),
            'Martel' => array( 200, 300, 400, 600, 700, 800, 900 ),
            'Martel Sans' => array( 200, 300, 400, 600, 700, 800, 900 ),
            'Marvel' => array( 400, 700 ),
            'Mate' => array( 400 ),
            'Mate SC' => array( 400 ),
            'Maven Pro' => array( 400, 500, 700, 900 ),
            'McLaren' => array( 400 ),
            'Meddon' => array( 400 ),
            'MedievalSharp' => array( 400 ),
            'Medula One' => array( 400 ),
            'Megrim' => array( 400 ),
            'Meie Script' => array( 400 ),
            'Merienda' => array( 400, 700 ),
            'Merienda One' => array( 400 ),
            'Merriweather' => array( 300, 400, 700, 900 ),
            'Merriweather Sans' => array( 300, 400, 700, 800 ),
            'Metal' => array( 400 ),
            'Metal Mania' => array( 400 ),
            'Metamorphous' => array( 400 ),
            'Metrophobic' => array( 400 ),
            'Michroma' => array( 400 ),
            'Milonga' => array( 400 ),
            'Miltonian' => array( 400 ),
            'Miltonian Tattoo' => array( 400 ),
            'Miniver' => array( 400 ),
            'Miss Fajardose' => array( 400 ),
            'Modak' => array( 400 ),
            'Modern Antiqua' => array( 400 ),
            'Molengo' => array( 400 ),
            'Molle' => array( 400 ),
            'Monda' => array( 400, 700 ),
            'Monofett' => array( 400 ),
            'Monoton' => array( 400 ),
            'Monsieur La Doulaise' => array( 400 ),
            'Montaga' => array( 400 ),
            'Montez' => array( 400 ),
            'Montserrat' => array( 400, 700 ),
            'Montserrat Alternates' => array( 400, 700 ),
            'Montserrat Subrayada' => array( 400, 700 ),
            'Moul' => array( 400 ),
            'Moulpali' => array( 400 ),
            'Mountains of Christmas' => array( 400, 700 ),
            'Mouse Memoirs' => array( 400 ),
            'Mr Bedfort' => array( 400 ),
            'Mr Dafoe' => array( 400 ),
            'Mr De Haviland' => array( 400 ),
            'Mrs Saint Delafield' => array( 400 ),
            'Mrs Sheppards' => array( 400 ),
            'Muli' => array( 300, 400 ),
            'Mystery Quest' => array( 400 ),
            'NTR' => array( 400 ),
            'Neucha' => array( 400 ),
            'Neuton' => array( 200, 300, 400, 700, 800 ),
            'New Rocker' => array( 400 ),
            'News Cycle' => array( 400, 700 ),
            'Niconne' => array( 400 ),
            'Nixie One' => array( 400 ),
            'Nobile' => array( 400, 700 ),
            'Nokora' => array( 400, 700 ),
            'Norican' => array( 400 ),
            'Nosifer' => array( 400 ),
            'Nothing You Could Do' => array( 400 ),
            'Noticia Text' => array( 400, 700 ),
            'Noto Sans' => array( 400, 700 ),
            'Noto Serif' => array( 400, 700 ),
            'Nova Cut' => array( 400 ),
            'Nova Flat' => array( 400 ),
            'Nova Mono' => array( 400 ),
            'Nova Oval' => array( 400 ),
            'Nova Round' => array( 400 ),
            'Nova Script' => array( 400 ),
            'Nova Slim' => array( 400 ),
            'Nova Square' => array( 400 ),
            'Numans' => array( 400 ),
            'Nunito' => array( 300, 400, 700 ),
            'Odor Mean Chey' => array( 400 ),
            'Offside' => array( 400 ),
            'Old Standard TT' => array( 400, 700 ),
            'Oldenburg' => array( 400 ),
            'Oleo Script' => array( 400, 700 ),
            'Oleo Script Swash Caps' => array( 400, 700 ),
            'Open Sans' => array( 300, 400, 600, 700, 800 ),
            'Open Sans Condensed' => array( 300, 700 ),
            'Oranienbaum' => array( 400 ),
            'Orbitron' => array( 400, 500, 700, 900 ),
            'Oregano' => array( 400 ),
            'Orienta' => array( 400 ),
            'Original Surfer' => array( 400 ),
            'Oswald' => array( 300, 400, 700 ),
            'Over the Rainbow' => array( 400 ),
            'Overlock' => array( 400, 700, 900 ),
            'Overlock SC' => array( 400 ),
            'Ovo' => array( 400 ),
            'Oxygen' => array( 300, 400, 700 ),
            'Oxygen Mono' => array( 400 ),
            'PT Mono' => array( 400 ),
            'PT Sans' => array( 400, 700 ),
            'PT Sans Caption' => array( 400, 700 ),
            'PT Sans Narrow' => array( 400, 700 ),
            'PT Serif' => array( 400, 700 ),
            'PT Serif Caption' => array( 400 ),
            'Pacifico' => array( 400 ),
            'Palanquin' => array( 100, 200, 300, 400, 500, 600, 700 ),
            'Palanquin Dark' => array( 400, 500, 600, 700 ),
            'Paprika' => array( 400 ),
            'Parisienne' => array( 400 ),
            'Passero One' => array( 400 ),
            'Passion One' => array( 400, 700, 900 ),
            'Pathway Gothic One' => array( 400 ),
            'Patrick Hand' => array( 400 ),
            'Patrick Hand SC' => array( 400 ),
            'Patua One' => array( 400 ),
            'Paytone One' => array( 400 ),
            'Peddana' => array( 400 ),
            'Peralta' => array( 400 ),
            'Permanent Marker' => array( 400 ),
            'Petit Formal Script' => array( 400 ),
            'Petrona' => array( 400 ),
            'Philosopher' => array( 400, 700 ),
            'Piedra' => array( 400 ),
            'Pinyon Script' => array( 400 ),
            'Pirata One' => array( 400 ),
            'Plaster' => array( 400 ),
            'Play' => array( 400, 700 ),
            'Playball' => array( 400 ),
            'Playfair Display' => array( 400, '400i', 700, '400i', 900, '900i' ),
            'Playfair Display SC' => array( 400, 700, 900 ),
            'Podkova' => array( 400, 700 ),
            'Poiret One' => array( 400 ),
            'Poller One' => array( 400 ),
            'Poly' => array( 400 ),
            'Pompiere' => array( 400 ),
            'Pontano Sans' => array( 400 ),
            'Poppins' => array( 300, 400, 500, 600, 700 ),
            'Port Lligat Sans' => array( 400 ),
            'Port Lligat Slab' => array( 400 ),
            'Pragati Narrow' => array( 400, 700 ),
            'Prata' => array( 400 ),
            'Preahvihear' => array( 400 ),
            'Press Start 2P' => array( 400 ),
            'Princess Sofia' => array( 400 ),
            'Prociono' => array( 400 ),
            'Prosto One' => array( 400 ),
            'Puritan' => array( 400, 700 ),
            'Purple Purse' => array( 400 ),
            'Quando' => array( 400 ),
            'Quantico' => array( 400, 700 ),
            'Quattrocento' => array( 400, 700 ),
            'Quattrocento Sans' => array( 400, 700 ),
            'Questrial' => array( 400 ),
            'Quicksand' => array( 300, 400, 700 ),
            'Quintessential' => array( 400 ),
            'Qwigley' => array( 400 ),
            'Racing Sans One' => array( 400 ),
            'Radley' => array( 400 ),
            'Rajdhani' => array( 300, 400, 500, 600, 700 ),
            'Raleway' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Raleway Dots' => array( 400 ),
            'Ramabhadra' => array( 400 ),
            'Ramaraja' => array( 400 ),
            'Rambla' => array( 400, 700 ),
            'Rammetto One' => array( 400 ),
            'Ranchers' => array( 400 ),
            'Rancho' => array( 400 ),
            'Ranga' => array( 400, 700 ),
            'Rationale' => array( 400 ),
            'Ravi Prakash' => array( 400 ),
            'Redressed' => array( 400 ),
            'Reenie Beanie' => array( 400 ),
            'Revalia' => array( 400 ),
            'Rhodium Libre' => array( 400 ),
            'Ribeye' => array( 400 ),
            'Ribeye Marrow' => array( 400 ),
            'Righteous' => array( 400 ),
            'Risque' => array( 400 ),
            'Roboto' => array( 100, 300, 400, 500, 700, 900 ),
            'Roboto Condensed' => array( 300, 400, 700 ),
            'Roboto Mono' => array( 100, 300, 400, 500, 700 ),
            'Roboto Slab' => array( 100, 300, 400, 700 ),
            'Rochester' => array( 400 ),
            'Rock Salt' => array( 400 ),
            'Rokkitt' => array( 400, 700 ),
            'Romanesco' => array( 400 ),
            'Ropa Sans' => array( 400 ),
            'Rosario' => array( 400, 700 ),
            'Rosarivo' => array( 400 ),
            'Rouge Script' => array( 400 ),
            'Rozha One' => array( 400 ),
            'Rubik' => array( 300, 400, 500, 700, 900 ),
            'Rubik Mono One' => array( 400 ),
            'Rubik One' => array( 400 ),
            'Ruda' => array( 400, 700, 900 ),
            'Rufina' => array( 400, 700 ),
            'Ruge Boogie' => array( 400 ),
            'Ruluko' => array( 400 ),
            'Rum Raisin' => array( 400 ),
            'Ruslan Display' => array( 400 ),
            'Russo One' => array( 400 ),
            'Ruthie' => array( 400 ),
            'Rye' => array( 400 ),
            'Sacramento' => array( 400 ),
            'Sahitya' => array( 400, 700 ),
            'Sail' => array( 400 ),
            'Salsa' => array( 400 ),
            'Sanchez' => array( 400 ),
            'Sancreek' => array( 400 ),
            'Sansita One' => array( 400 ),
            'Sarala' => array( 400, 700 ),
            'Sarina' => array( 400 ),
            'Sarpanch' => array( 400, 500, 600, 700, 800, 900 ),
            'Satisfy' => array( 400 ),
            'Scada' => array( 400, 700 ),
            'Scheherazade' => array( 400, 700 ),
            'Schoolbell' => array( 400 ),
            'Seaweed Script' => array( 400 ),
            'Sevillana' => array( 400 ),
            'Seymour One' => array( 400 ),
            'Shadows Into Light' => array( 400 ),
            'Shadows Into Light Two' => array( 400 ),
            'Shanti' => array( 400 ),
            'Share' => array( 400, 700 ),
            'Share Tech' => array( 400 ),
            'Share Tech Mono' => array( 400 ),
            'Shojumaru' => array( 400 ),
            'Short Stack' => array( 400 ),
            'Siemreap' => array( 400 ),
            'Sigmar One' => array( 400 ),
            'Signika' => array( 300, 400, 600, 700 ),
            'Signika Negative' => array( 300, 400, 600, 700 ),
            'Simonetta' => array( 400, 900 ),
            'Sintony' => array( 400, 700 ),
            'Sirin Stencil' => array( 400 ),
            'Six Caps' => array( 400 ),
            'Skranji' => array( 400, 700 ),
            'Slabo 13px' => array( 400 ),
            'Slabo 27px' => array( 400 ),
            'Slackey' => array( 400 ),
            'Smokum' => array( 400 ),
            'Smythe' => array( 400 ),
            'Sniglet' => array( 400, 800 ),
            'Snippet' => array( 400 ),
            'Snowburst One' => array( 400 ),
            'Sofadi One' => array( 400 ),
            'Sofia' => array( 400 ),
            'Sonsie One' => array( 400 ),
            'Sorts Mill Goudy' => array( 400 ),
            'Source Code Pro' => array( 200, 300, 400, 500, 600, 700, 900 ),
            'Source Sans Pro' => array( 200, 300, 400, 600, 700, 900 ),
            'Source Serif Pro' => array( 400, 600, 700 ),
            'Special Elite' => array( 400 ),
            'Spicy Rice' => array( 400 ),
            'Spinnaker' => array( 400 ),
            'Spirax' => array( 400 ),
            'Squada One' => array( 400 ),
            'Sree Krushnadevaraya' => array( 400 ),
            'Stalemate' => array( 400 ),
            'Stalinist One' => array( 400 ),
            'Stardos Stencil' => array( 400, 700 ),
            'Stint Ultra Condensed' => array( 400 ),
            'Stint Ultra Expanded' => array( 400 ),
            'Stoke' => array( 300, 400 ),
            'Strait' => array( 400 ),
            'Sue Ellen Francisco' => array( 400 ),
            'Sumana' => array( 400, 700 ),
            'Sunshiney' => array( 400 ),
            'Supermercado One' => array( 400 ),
            'Sura' => array( 400, 700 ),
            'Suranna' => array( 400 ),
            'Suravaram' => array( 400 ),
            'Suwannaphum' => array( 400 ),
            'Swanky and Moo Moo' => array( 400 ),
            'Syncopate' => array( 400, 700 ),
            'Tangerine' => array( 400, 700 ),
            'Taprom' => array( 400 ),
            'Tauri' => array( 400 ),
            'Teko' => array( 300, 400, 500, 600, 700 ),
            'Telex' => array( 400 ),
            'Tenali Ramakrishna' => array( 400 ),
            'Tenor Sans' => array( 400 ),
            'Text Me One' => array( 400 ),
            'The Girl Next Door' => array( 400 ),
            'Tienne' => array( 400, 700, 900 ),
            'Tillana' => array( 400, 500, 600, 700, 800 ),
            'Timmana' => array( 400 ),
            'Tinos' => array( 400, 700 ),
            'Titan One' => array( 400 ),
            'Titillium Web' => array( 200, 300, 400, 600, 700, 900 ),
            'Trade Winds' => array( 400 ),
            'Trocchi' => array( 400 ),
            'Trochut' => array( 400, 700 ),
            'Trykker' => array( 400 ),
            'Tulpen One' => array( 400 ),
            'Ubuntu' => array( 300, 400, 500, 700 ),
            'Ubuntu Condensed' => array( 400 ),
            'Ubuntu Mono' => array( 400, 700 ),
            'Ultra' => array( 400 ),
            'Uncial Antiqua' => array( 400 ),
            'Underdog' => array( 400 ),
            'Unica One' => array( 400 ),
            'UnifrakturCook' => array( 700 ),
            'UnifrakturMaguntia' => array( 400 ),
            'Unkempt' => array( 400, 700 ),
            'Unlock' => array( 400 ),
            'Unna' => array( 400 ),
            'VT323' => array( 400 ),
            'Vampiro One' => array( 400 ),
            'Varela' => array( 400 ),
            'Varela Round' => array( 400 ),
            'Vast Shadow' => array( 400 ),
            'Vesper Libre' => array( 400, 500, 700, 900 ),
            'Vibur' => array( 400 ),
            'Vidaloka' => array( 400 ),
            'Viga' => array( 400 ),
            'Voces' => array( 400 ),
            'Volkhov' => array( 400, 700 ),
            'Vollkorn' => array( 400, 700 ),
            'Voltaire' => array( 400 ),
            'Waiting for the Sunrise' => array( 400 ),
            'Wallpoet' => array( 400 ),
            'Walter Turncoat' => array( 400 ),
            'Warnes' => array( 400 ),
            'Wellfleet' => array( 400 ),
            'Wendy One' => array( 400 ),
            'Wire One' => array( 400 ),
            'Work Sans' => array( 100, 200, 300, 400, 500, 600, 700, 800, 900 ),
            'Yanone Kaffeesatz' => array( 200, 300, 400, 700 ),
            'Yantramanav' => array( 100, 300, 400, 500, 700, 900 ),
            'Yellowtail' => array( 400 ),
            'Yeseva One' => array( 400 ),
            'Yesteryear' => array( 400 ),
            'Zeyada' => array( 400 ),
        );
    }
    /**
     * Display navigation to next/previous set of posts when applicable.
     */
    public static function udetor_paging_nav() {
        // Don't print empty markup if there's only one page.
        if ( $GLOBALS['wp_query']->max_num_pages < 2 ) {
            return;
        }

        $paged        = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
        $pagenum_link = html_entity_decode( get_pagenum_link() );
        $query_args   = array();
        $url_parts    = explode( '?', $pagenum_link );

        if ( isset ( $url_parts[1] ) ) {
            wp_parse_str( $url_parts[1], $query_args );
        }

        $pagenum_link = remove_query_arg( array_keys( $query_args ), $pagenum_link );
        $pagenum_link = trailingslashit( $pagenum_link ) . '%_%';

        $format  = $GLOBALS['wp_rewrite']->using_index_permalinks() && ! strpos( $pagenum_link, 'index.php' ) ? 'index.php/' : '';
        $format .= $GLOBALS['wp_rewrite']->using_permalinks() ? user_trailingslashit( 'page/%#%', 'paged' ) : '?paged=%#%';

        // Set up paginated links.
        $links = paginate_links( array(
            'udetor'     => $pagenum_link,
            'format'    => $format,
            'total'     => $GLOBALS['wp_query']->max_num_pages,
            'current'   => $paged,
            'mid_size'  => 1,
            'add_args'  => array_map( 'urlencode', $query_args ),
            'prev_text' =>  wp_kses(__('<i class=\'fa fa-arrow-left\'></i>', 'udetor' ), array('i' => array('class' => array()))),
            'next_text' => wp_kses(__( '<i class=\'fa fa-arrow-right\'></i>', 'udetor' ), array('i' => array('class' => array()))),
        ) );

        if ( $links ) :
        ?>
        <nav class="udetor-paging-navigation">
            <h2 class="screen-reader-text"><?php esc_html_e( 'Posts navigation', 'udetor' ); ?></h2>
            <div class="pagination loop-pagination">
                <?php echo wp_kses($links, array(
                    'a' => array(
                        'href' => array(),
                        'class' => array()
                    ),
                    'i' => array(
                        'class' => array()
                    ),
                    'span' => array(
                        'class' => array()
                    )
                )); ?>
            </div><!--/ .pagination -->
        </nav><!--/ .navigation -->
        <?php
        endif;
    }
    /**
     * Render page loader.
     *
     * @return  string
     */
    public static function udetor_page_loader( $output = NULL ) {
        $type = get_theme_mod( 'udetor_effect_type', 'none' );        

        // Loader effects css
        $effects_css = get_theme_mod( 'udetor_css_animation_type', '1' );

        // Loader effects css
        $effects_image = get_theme_mod( 'udetor_page_effect_image','');

        if ( 'css' == $type ) {
            $output .= '<div class="pageloader">';            
            $output .= '<div class="udetor-loader-' . esc_attr( $effects_css ) . '"><div class="udetor-loader"></div></div>';
            

            $output .= '</div>';
        } elseif ( 'image' == $type && ! empty( $effects_image ) ) {
            $output .= '<div class="pageloader"><div class="loader-img">';
            $output .= '<img src="' . esc_url( $effects_image ) . '" />';
            $output .= '</div></div>';
        }

        return apply_filters( 'udetor_page_loader', $output );
    }

    public static function udetor_build_anchor_links( $permalink, $title ) {

        return sprintf(
            '<span itemscope="" itemtype="http://schema.org/Breadcrumb"><a href="%1$s" title="%2$s" rel="%3$s" class="trail-begin"><span itemprop="%2$s">%4$s</span></a></span>',
            esc_url( $permalink ),
            esc_attr( $title ),
            sanitize_title( $title ),
            esc_html( $title )
        );

    }

    /**
     * @param        $post_id
     * @param string $taxonomy
     *
     * @return bool
     */
    public static function udetor_post_taxonomy( $post_id, $taxonomy = 'category' ) {
        $terms = get_the_terms( $post_id, $taxonomy );
        $t     = array();
        if ( $terms ) {
            foreach ( $terms as $term ) {
                $t[] = self::udetor_build_anchor_links( get_term_link( $term->slug, $taxonomy ), $term->name );
            }

            return implode( ' / ', $t );
        } else {
            return false;
        }
    }
    

    public static function udetor_learn_dash_breadcrumbs() {
        global $wp_query;        
        
        $course_label      = 'courses' ;
        // Define main variables
        $trail   = array();
        $trail[] = self::udetor_build_anchor_links( esc_url( home_url( '/' ) ), esc_html__( 'Home', 'udetor' ) );
        
        $dashboard_link      = '';
        $dashboard_text      = 'Dashboard';
        $dashboard_separator ='/';        
        
        $course_archive_link     = self::udetor_build_anchor_links( get_post_type_archive_link( 'sfwd-courses' ), esc_html( $course_label ) );
               
        $lesson_id = false;

        // If it's on home page
        if ( is_front_page() ) {
            $trail = array(); //Removing Single Home link from Homepage.
        } elseif ( is_singular() ) {
            // Get singular vars (page, post, attachments)
            $post      = $wp_query->get_queried_object();
            $post_id   = absint( $wp_query->get_queried_object_id() );
            $post_type = $post->post_type;

            if ( 'post' === $post_type ) {
                $maybe_tax = self::udetor_learn_post_taxonomy( $post_id );

                if ( false !== $maybe_tax ) {
                    $trail[] = $maybe_tax;
                }
                $trail[] = get_the_title( $post_id );

            } elseif ( 'page' === $post_type ) {
                
                    // Regular pages. See if the page has any ancestors. Add in the trail if ancestors are found
                    $ancestors = get_ancestors( $post_id, 'page' );
                    if ( ! empty ( $ancestors ) ) {
                        $ancestors = array_reverse( $ancestors );
                        foreach ( $ancestors as $page ) {
                            $trail[] = self::udetor_build_anchor_links( get_permalink( $page ), get_the_title( $page ) );
                        }
                    }
                
            } elseif ( 'sfwd-courses' === $post_type ) {
                // See if Single Course is being displayed.
                
                $trail[] = $course_archive_link;
                
                $trail[] = get_the_title( $post_id );
            } elseif ( 'sfwd-lessons' === $post_type ) {
                // See if Single Lesson is being displayed.
                $course_id = get_post_meta( $post_id, 'course_id', true ); // Getting Parent Course ID
               
                    $trail[] = $course_archive_link;
               
                $trail[] = self::udetor_build_anchor_links( get_permalink( $course_id ), get_the_title( $course_id ) ); // Getting Lesson's Course Link
                
            } elseif ( 'sfwd-topic' === $post_type ) {
                // See if single Topic is being displayed
                $course_id = get_post_meta( $post_id, 'course_id', true ); // Getting Parent Course ID
                $lesson_id = get_post_meta( $post_id, 'lesson_id', true ); // Getting Parent Lesson ID
                
                    $trail[] = $course_archive_link;
                
                $trail[] = self::udetor_build_anchor_links( get_permalink( $course_id ), get_the_title( $course_id ) ); // Getting Lesson's Course Link
                $trail[] = self::udetor_build_anchor_links( get_permalink( $lesson_id ), get_the_title( $lesson_id ) ); // Getting Topics's Lesson Link
                
            } elseif ( 'sfwd-quiz' === $post_type ) {
                // See if quiz is being displayed
                $course_id = get_post_meta( $post_id, 'course_id', true ); // Getting Parent Course ID
                    $trail[] = $course_archive_link;
                

                $topic_id = get_post_meta( $post_id, 'lesson_id', true ); // Getting Parent Topic/Lesson ID
                if ( 'sfwd-topic' === get_post_type( $topic_id ) ) {
                    $lesson_id = get_post_meta( $topic_id, 'lesson_id', true ); // Getting Parent Lesson ID
                }
                $trail[] = self::udetor_build_anchor_links( get_permalink( $course_id ), get_the_title( $course_id ) ); // Getting Lesson's Course Link
                //If $lesson_id is false, the quiz is associated with a lesson and course but not a topic.
                if ( $lesson_id ) {
                    $trail[] = self::udetor_build_anchor_links( get_permalink( $lesson_id ), get_the_title( $lesson_id ) ); // Getting Topics's Lesson Link
                }
                //If $topic_id is false, the quiz is associated with a course but not associated with any lessons or topics.
                if ( $topic_id ) {
                    $trail[] = self::udetor_build_anchor_links( get_permalink( $topic_id ), get_the_title( $topic_id ) );
                }
                

            } else {
                

                // Getting terms of the post.
                if ( self::udetor_lms_get_taxonomy( $post_id, $post_type ) ) {
                    $trail[] = self::udetor_lms_get_taxonomy( $post_id, $post_type );
                }
                
            }
        }
        // If it's an Archive
        if ( is_archive() ) {
            //Ignore if Courses & Products
            if ( ! is_post_type_archive( 'sfwd-courses' ) && ! is_post_type_archive( 'product' ) ) {
                if ( is_category() || is_tax() ) {
                    $trail[] = single_cat_title( '', false ); // If its Blog Category
                }
                if ( is_day() ) {
                    $trail[] = get_the_date(); // If its Single Day Archive
                }
                if ( is_month() ) {
                    $trail[] = get_the_date( __( 'F Y', 'udetor' ) ) . esc_html__( ' Archives', 'udetor' ); // If Mothly Archives
                }
                if ( is_year() ) {
                    $trail[] = get_the_date( __( 'Y', 'udetor' ) ) . esc_html__( ' Archives', 'udetor' ); // If its Yearly Archives
                }
                if ( is_author() ) {
                    $trail[] = get_the_author(); // If its Author's Archives
                }
            } elseif ( is_post_type_archive( 'sfwd-courses' ) ) {
                $trail[] = esc_html( $course_label );
            } elseif ( is_post_type_archive( 'product' ) ) {
                $trail[] = esc_html__( 'Shop', 'udetor' );
            }
        }

        if ( is_search() ) {
            $trail[] = esc_html__( 'Search', 'udetor' );
            $trail[] = get_search_query();
        }

        // Build breadcrumbs
        $classes = 'sfwd-breadcrumbs clr';

        if ( array_key_exists( 'the_content', $GLOBALS['wp_filter'] ) ) {
            $classes .= ' lms-breadcrumbs ';
        }

        // Open breadcrumbs
        $breadcrumb = '<nav class="' . esc_attr( $classes ) . '"><div class="breadcrumb-trail">';

        // Separator HTML
        $separator = '<span class="sep"> ' . stripslashes( $dashboard_separator ) . ' </span>';

        // Join all trail items into a string
        $breadcrumb .= implode( $separator, $trail );

        // Close breadcrumbs
        $breadcrumb .= '</div></nav>';

        return $breadcrumb;
    }

    public static function udetor_learn_post_taxonomy( $post_id, $taxonomy = 'category' ) {
        $terms = get_the_terms( $post_id, $taxonomy );
        $t     = array();
        if ( $terms ) {
            foreach ( $terms as $term ) {
                $t[] = self::udetor_build_anchor_links( get_term_link( $term->slug, $taxonomy ), $term->name );
            }

            return implode( ' / ', $t );
        } else {
            return false;
        }
    }
    public static function udetor_lms_get_taxonomy( $post_id, $post_type ) {
        $taxonomies = get_taxonomies( array( 'object_type' => array( $post_type ) ), 'objects' );
        $tax        = array();
        if ( $taxonomies ) {
            foreach ( $taxonomies as $taxonomy ) {
                // Pass the $taxonomy name to udetor_learn_post_taxonomy to return with proper terms and links
                $tax[] = self::udetor_learn_post_taxonomy( $post_id, $taxonomy->query_var );
            }

            return implode( ' / ', $tax );
        } else {
            return false;
        }
    }

    public static function udetor_reviews_star( $postid ) {
        
        $args = array(
        'post_id' => $postid,
        'status' => 'approve',             
        );
        $comments = get_comments( $args );
        $ratings = array();
        $output='';
        if ( is_plugin_active( 'netbase-toolkit/app.php' ) ) {
            foreach ($comments as $comment) {
                $star = get_comment_meta( $comment->comment_ID, 'nbcsr_rating', true);                    
                    if( !empty( $star ) ) {
                        array_push( $ratings, $star );
                    }
            }

            if(!empty($ratings)){
                $total      = array_sum($ratings);
                $count      = count($ratings);        
                if( $count > 0 ) {
                    $average    = $total / $count;
                    $ratings_arrange = array_count_values($ratings);
                }
                
                // Insert 0 if it is undefined or empty
                for ( $i = 1; $i <= 5; $i++) {
                    if( !isset($ratings_arrange[$i]) ) {
                        $ratings_arrange[$i] = 0;
                    }
                }
                $output = wp_star_rating( array(
                    'rating'    => $average,
                    'type'      => 'rating',
                    'number'    => $count,
                    'echo'      => false
                ));
            }else{
                $output = wp_star_rating( array(                
                    'type'      => 'rating',                
                    'echo'      => false
                ));
            }
        }
        //var_dump($average);

        return $output;
        
    }   
    /**
     * Get infomation course session topic for the theme.
     *
     * @return string for the theme.
     */
    public static function udetor_time_second( $time ) {
        $second = 0;
        if ( preg_match( '/^(?:(?<hours>\d+)h\s*)?(?:(?<minutes>\d+)m\s*)?(?:(?<seconds>\d+)s\s*)?$/', $time, $matches ) ) {
            $second = ( ! empty( $matches['hours'] ) ? $matches['hours'] * 3600 : '00' );
            $second += ( ! empty( $matches['minutes'] ) ? $matches['minutes'] * 60 : '00' );
            $second += ( ! empty( $matches['seconds'] ) ? $matches['seconds'] : '00' );
        }
        if ( preg_match( '/^(?:(?<hours>\d+)hr\s*)?(?:(?<minutes>\d+)min\s*)?(?:(?<seconds>\d+)sec\s*)?$/', $time, $matches ) ) {
            $second = ( ! empty( $matches['hours'] ) ? $matches['hours'] * 3600 : '00' );
            $second += ( ! empty( $matches['minutes'] ) ? $matches['minutes'] * 60 : '00' );
            $second += ( ! empty( $matches['seconds'] ) ? $matches['seconds'] : '00' );
        }

        return $second;
    }

    public static function udetor_duration_course( $course_id ) {
        $total_second = 0;
        $course       = get_post( $course_id );
        $lessons      = learndash_get_course_lessons_list( $course );
        if ( is_array( $lessons ) || is_object( $lessons ) ) :
            foreach ( $lessons as $lesson ) :
                $meta = get_post_meta( $lesson['post']->ID, '_' . $lesson['post']->post_type );
                if ( $meta[0][ $lesson['post']->post_type . '_forced_lesson_time' ] ) :
                    $time = $meta[0][ $lesson['post']->post_type . '_forced_lesson_time' ];
                    $total_second += self::udetor_time_second( $time );
                    $total_time = number_format( $total_second / 3600, 1 );
                else :
                    $lesson_topics[ $lesson['post']->ID ] = learndash_topic_dots( $lesson['post']->ID, false, 'array' );
                    $topics                               = $lesson_topics[ $lesson['post']->ID ];
                    if ( is_array( $topics ) || is_object( $topics ) ) :
                        foreach ( (array) $topics as $key => $topic ) :
                            $meta = get_post_meta( $topic->ID, '_' . $topic->post_type );
                            if ( $meta[0][ $topic->post_type . '_forced_lesson_time' ] ) :
                                $time = $meta[0][ $topic->post_type . '_forced_lesson_time' ];
                                $total_second += self::udetor_time_second( $time );
                            endif;
                        endforeach;
                    endif;
                endif;
            endforeach;
        endif;
        $total_time = number_format( $total_second / 3600, 1 );
        return $total_time;
    }
}