<?php
/**
 * Plug additional sidebars into Admin.
 *
 * @package  Udetor
 * @since    1.0
 */
class Udetor_HeaderDesigner
{
    /**
     * Variable to hold the initialization state.
     *
     * @var  boolean
     */
    protected static $initialized = false;

    /**
     * Initialize functions.
     *
     * @return  void
     */
    public static function initialize()
    {
        // Do nothing if start functions already initialized.
        if (self::$initialized) {
            return;
        }
        
        add_action('wp_before_admin_bar_render', array(__CLASS__, 'edit_header_in_bar'));

        // Register actions to do something.
        add_action( 'vc_after_init', array( __CLASS__, 'vc_after_init_actions' ));
        add_action( 'wp_head', array( __CLASS__, 'add_function_insert_internal_css' ) );

        // State that initialization completed.
        self::$initialized = true;
    }


   /**
     * Add script function insert internal css.
     *
     * @since 1.0.1
     */
    public static function add_function_insert_internal_css() {
        ?>
        <script type="text/javascript">
            function udetor_insert_internal_css(css) {
                var udetor_style = document.createElement("style");
                udetor_style.type = "text/css";
                udetor_style.setAttribute('data-type', 'udetor-internal-css');
                var udetor_style_content = document.createTextNode(css);
                udetor_style.appendChild(udetor_style_content);
                document.head.appendChild(udetor_style);
            }
        </script>
        <?php
    }

    public static function render_content( $content, $builder) {
        
        $rendered = do_shortcode( $content );

        if ( $builder == 'vc' && function_exists( 'visual_composer' ) ) {
            $custom_css = visual_composer()->parseShortcodesCustomCss( $content );

            if ( ! empty( $custom_css ) ) {
                $script = '<script type="text/javascript">
                udetor_insert_internal_css("' . $custom_css . '");
            </script>';

                $rendered .= $script;
            }
        }

        return $rendered;
    }

    /**
     * Register page type for visual composer
     * After VC Init
     */
    public static function vc_after_init_actions() {

        // Enable VC by default on a list of Post Types
        if( function_exists('vc_set_default_editor_post_types') ){
            $list = array(
                'page',
                'page_header', // add here your custom post types slug
            );
            vc_set_default_editor_post_types( $list );
        }

        // Disable AdminBar VC edit link
        if( function_exists('vc_frontend_editor') ){
            remove_action( 'admin_bar_menu', array( vc_frontend_editor(), 'adminBarEditLink' ), 1000 );
        }
        
    }

    /**
     * Add Header Presets
     */
    public static function add_header_presets()
    {
        /* Header Preset */
        Kirki::add_section('custom_header', array(
            'title' => esc_html__('Header','udetor'),
            'panel' => '', // Not typically needed.
            'priority' => 160,
            'capability' => 'edit_theme_options',
            'theme_supports' => '', // Rarely needed.
        ));

        // Get all headers.
        $list_header = new WP_Query(array(
            'posts_per_page' => -1,
            'post_type' => 'page_header',
            'post_status' => 'publish',
            'suppress_filters' => true,
        ));

        $header_layout = array();

        // Set to normal headers default
        if ($list_header->post_count) {
            foreach ($list_header->posts as $val) {
                $image = get_the_post_thumbnail($val->ID, 'thumbnail');
                $post_thumbnail_id = get_post_thumbnail_id($val->ID);
                $image_url = wp_get_attachment_image_url($post_thumbnail_id, 'thumbnail');
                $header_layout[$val->ID] = $image_url;
            }
        };

        Kirki::add_field('my_config', array(
            'type' => 'radio-image',
            'settings' => 'udetor_global_header',
            'label' => esc_html__('Header Presets', 'udetor'),
            'section' => 'udetor_header',
            'priority' => 10,
            'choices' => $header_layout,
        ));
    }    

    /**
     * Add a note edit header in bar
     *
     * @return  void
     */
    public static function edit_header_in_bar()
    {
        if (!is_super_admin()) return;

        global $wp_admin_bar;

        $current_header = get_theme_mod('udetor_global_header');

        if ($current_header > 0) {
            $wp_admin_bar->add_node(array(
                'id' => 'edit-header',
                'title' => esc_html__('Edit Header', 'udetor'),
                'href' => admin_url() . 'post.php?post=' . $current_header . '&action=edit'
            ));
        } 

    }

}