<?php
// Prevent direct access to this file
defined( 'ABSPATH' ) || die( 'Direct access to this file is not allowed.' );

/**
 * Core class.
 *
 * @package  Udetor_Theme
 * @since    1.0
 */
class Udetor_Core{
    /**
     * Define theme version.
     *
     * @var  string
     */
    const VERSION = '1.0.0';
    
    /**
     * Define valid class prefix for autoloading.
     *
     * @var  string
     */
    protected static $prefix = 'Udetor_';
    
    /**
     * Initialize Udetor_Theme.
     *
     * @return  void
     */
    public static function initialize() {

        require_once get_template_directory() . '/inc/class-tgm-plugin-activation.php';

        require_once get_template_directory() . '/inc/modules/ajax-search/ajax-search.php';

        // Register class autoloader.
        spl_autoload_register( array( __CLASS__, 'autoload' ) );
        
        // Include function plugins if not include.
        self::include_function_plugins();

        // Add custom css to admin
        add_action( 'admin_head', array( __CLASS__, 'admin_custom_css' ) );

        add_action('admin_enqueue_scripts', array( __CLASS__, 'redirect_merlin') );

        add_action('after_switch_theme', array( __CLASS__, 'udetor_setup_del_options') );
        
        // Plug into Site and supported 3rd-party plugins.
        Udetor_Start::initialize();
    }

    public static function udetor_setup_del_options() {
        delete_option( 'merlin_udetor_redirected' );
    }

    public static function redirect_merlin() {
        wp_register_script( 'redirect-page', get_template_directory_uri() . '/assets/udetor/js/admin/redirect-page.js', array('jquery'), time() );

        $translation_array = array(
            'import_merlin_url' => admin_url( 'themes.php?page=merlin' )
        );
        wp_localize_script( 'redirect-page', 'object_merlin', $translation_array );

        wp_enqueue_script( 'redirect-page' );
    }
    
    /**
     * Method to autoload class declaration file.
     *
     * @param   string  $class_name  Name of class to load declaration file for.
     *
     * @return  mixed
     */
    public static function autoload( $class_name ) {
        // Verify class prefix.
        if ( 0 !== strpos( $class_name, self::$prefix ) ) {
            return false;
        }
        
        // Generate file path from class name.
		$base = get_parent_theme_file_path() . '/udetor/includes/';
        $path = strtolower( str_replace( '_', '/', substr( $class_name, strlen( self::$prefix ) ) ) );
        
        // Check if class file exists.
        $standard    = $path . '.php';
        $alternative = $path . '/' . current( array_slice( explode( '/', str_replace( '\\', '/', $path ) ), -1 ) ) . '.php';
        
        while ( true ) {
            // Check if file exists in standard path.
            if ( @is_file( $base . $standard ) ) {
                $exists = $standard;
                
                break;
            }
            
            // Check if file exists in alternative path.
            if ( @is_file( $base . $alternative ) ) {
                $exists = $alternative;
                
                break;
            }
            
            // If there is no more alternative file, quit the loop.
            if ( false === strrpos( $standard, '/' ) || 0 === strrpos( $standard, '/' ) ) {
                break;
            }
            
            // Generate more alternative files.
            $standard    = preg_replace( '#/([^/]+)$#', '-\\1', $standard );
            $alternative = implode( '/', array_slice( explode( '/', str_replace( '\\', '/', $standard ) ), 0, -1 ) ) . '/' . substr( current( array_slice( explode( '/', str_replace( '\\', '/', $standard ) ), -1 ) ), 0, -4 ) . '/' . current( array_slice( explode( '/', str_replace( '\\', '/', $standard ) ), -1 ) );
        }
        
        // Include class declaration file if exists.
        if ( isset( $exists ) ) {
            return include_once $base . $exists;
        }
        
        return false;
    }
    
    /**
     * Include function plugins if not include.
     *
     * @return  void
     *
     * @since  1.1.8
     *
     */
    public static function include_function_plugins() {
        if ( ! function_exists( 'is_plugin_active' ) ) {
            require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
        }
        
        if ( ! function_exists( 'wp_get_current_user' ) ) {
            require_once( ABSPATH . 'wp-includes/pluggable.php' );
        }
    
        if ( ! class_exists( 'WP_Customize_Section' ) ) {
            require_once( ABSPATH . 'wp-includes/class-wp-customize-section.php' );
        }
    }
    
    /**
     * Creates a new top level menu section.
     *
     * @return  void
     */
    public static function admin_menu() {
        
    }
    
    /**
     * Add some custom css to control notice of 3rd-plugin
     *
     * @since  1.1.8
     */
    public static function admin_custom_css() {
        echo '
			<style>
				.vc_license-activation-notice.updated,
				.rs-update-notice-wrap.updated,
				.installer-q-icon {
					display: none;
				}
			</style>
		';
    }
	
	/**
	 * Get Libs Path - Autoload classes
	 */
	public static function get_libs_path(){
		return trailingslashit(get_parent_theme_file_path()) . 'inc/libs/';
	}

    /**
     * Get Libs Path URL
     */
    public static function get_libs_path_url(){
        return trailingslashit(get_parent_theme_file_uri()) . 'inc/libs/';
    }

	/**
	 * Get Plugins Path - Autoload classes
	 */
	public static function get_plugins_path(){
		return trailingslashit(get_parent_theme_file_path()) . 'inc/plugins/';
	}
    
    /**
     * Get Core Path - Autoload classes
    */
    public static function get_core_path(){
        return trailingslashit(get_parent_theme_file_path()) . 'udetor/includes/';
    }
    
    /**
     * Get Core Path - Autoload classes
     */
    public static function get_core_path_url(){
        return trailingslashit(get_parent_theme_file_uri()) . 'udetor/includes/';
    }
    
    /**
     * Get Core Templates Path
     */
    public static function get_core_templates_path(){
        return trailingslashit(get_parent_theme_file_path()) . 'udetor/templates/';
    }

    public static function get_udetor_template( $tmpl, $extension = NULL ) {
        get_template_part( 'udetor/templates/' . $tmpl, $extension );
    }    
    
}

// Initialize PREFIX.
Udetor_CORE::initialize();