<?php
/**
 * @version    1.0.0
 * @package    Netbase_Toolkit
 * @author     Netbase Team <support@netbaseteam.com>
 * @copyright  Copyright (C) 2014 NetbaseTeam.com. All Rights Reserved.
 * @license    GNU/GPL v2 or later http://www.gnu.org/licenses/gpl-2.0.html
 *
 * Websites: http://www.netbaseteam.com
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) )
    exit;

/**
 * Define class to register and manage custom post type netbase toolkit.
 */
class Netbase_Toolkit_Post_Type {

    /**
     * Define plugin uri.
     *
     * @var  string
     */
    public static $plugin_uri;

	/**
	 * Initialize.
	 *
	 * @return  void
	 */
	public static function initialize() {

	    // Post Type with slash character
	    self::$plugin_uri = trailingslashit(NETBASE_TOOLKIT_URL);

		// Register post type
		//add_action( 'init', array( __CLASS__, 'register' ) );

		// Custom template for single & archive toolkit
		add_filter( 'single_template', array( __CLASS__, 'single_template' ) );
		add_filter( 'archive_template', array( __CLASS__, 'archive_template' ) );

		// Enqueue script
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );

		// Customize toolkit message
		add_filter( 'post_updated_messages', array( __CLASS__, 'updated_messages' ) );
	}


   /**
	 * Register netbase toolkit post types.
	 *
	 *
	 * @return  string
	 */
	public static function register() {
		$slug = get_theme_mod( 'toolkit_slug', 'toolkit' );

		register_post_type(
			'netbase-toolkit',
			array(
				'labels' => array(
					'name'               => __( 'Netbase Toolkit', 'netbase-toolkit' ),
					'singular_name'      => __( 'Toolkit', 'netbase-toolkit' ),
					'menu_name'          => __( 'NB Toolkit', 'netbase-toolkit' ),
					'name_admin_bar'     => __( 'Toolkit', 'netbase-toolkit' ),
					'add_new'            => __( 'Add New', 'netbase-toolkit' ),
					'add_new_item'       => __( 'Add New Toolkit', 'netbase-toolkit' ),
					'new_item'           => __( 'New Toolkit', 'netbase-toolkit' ),
					'edit_item'          => __( 'Edit Toolkit', 'netbase-toolkit' ),
					'view_item'          => __( 'View Toolkit', 'netbase-toolkit' ),
					'all_items'          => __( 'All Toolkit', 'netbase-toolkit' ),
					'search_items'       => __( 'Search Toolkit', 'netbase-toolkit' ),
					'parent_item_colon'  => __( 'Parent Toolkit:', 'netbase-toolkit' ),
					'not_found'          => __( 'No toolkit found.', 'netbase-toolkit' ),
					'not_found_in_trash' => __( 'No toolkit found in Trash.', 'netbase-toolkit' )
				),
				'public'             => true,
				'publicly_queryable' => true,
				'show_ui'            => true,
				'show_in_menu'       => true,
				'query_var'          => true,
				'rewrite'            => array( 'slug' => esc_attr( $slug ) ),
				'capability_type'    => 'post',
				'has_archive'        => true,
				'hierarchical'       => false,
				'menu_position'      => null,
				'menu_icon'          => 'dashicons-editor-help',
				'supports'           => array( 'title', 'editor', 'thumbnail' )
			)
		);

		// Add new taxonomy, make it hierarchical (like categories)
		register_taxonomy(
			'toolkit_cat',
			'netbase-toolkit',
			array(
				'labels' => array(
					'name'              => __( 'Categories', 'netbase-toolkit' ),
					'singular_name'     => __( 'Category', 'netbase-toolkit' ),
					'search_items'      => __( 'Search Categories', 'netbase-toolkit' ),
					'all_items'         => __( 'All Categories', 'netbase-toolkit' ),
					'parent_item'       => __( 'Parent Category', 'netbase-toolkit' ),
					'parent_item_colon' => __( 'Parent Category:', 'netbase-toolkit' ),
					'edit_item'         => __( 'Edit Category', 'netbase-toolkit' ),
					'update_item'       => __( 'Update Category', 'netbase-toolkit' ),
					'add_new_item'      => __( 'Add New Category', 'netbase-toolkit' ),
					'new_item_name'     => __( 'New Category Name', 'netbase-toolkit' ),
					'menu_name'         => __( 'Categories', 'netbase-toolkit' ),
				),
				'hierarchical'      => true,
				'show_ui'           => true,
				'show_admin_column' => true,
				'query_var'         => true,
				'rewrite'           => array( 'slug' => 'toolkit_cat' ),
			)
		);

		// Add new taxonomy, NOT hierarchical (like tags)
		register_taxonomy(
			'toolkit_tag',
			'netbase-toolkit',
			array(
				'labels' => array(
					'name'                       => __( 'Tags', 'netbase-toolkit' ),
					'singular_name'              => __( 'Tag', 'netbase-toolkit' ),
					'search_items'               => __( 'Search Tags', 'netbase-toolkit' ),
					'popular_items'              => __( 'Popular Tags', 'netbase-toolkit' ),
					'all_items'                  => __( 'All Tags', 'netbase-toolkit' ),
					'parent_item'                => null,
					'parent_item_colon'          => null,
					'edit_item'                  => __( 'Edit Tag', 'netbase-toolkit' ),
					'update_item'                => __( 'Update Tag', 'netbase-toolkit' ),
					'add_new_item'               => __( 'Add New Tag', 'netbase-toolkit' ),
					'new_item_name'              => __( 'New Tag Name', 'netbase-toolkit' ),
					'separate_items_with_commas' => __( 'Separate writers with commas', 'netbase-toolkit' ),
					'add_or_remove_items'        => __( 'Add or remove writers', 'netbase-toolkit' ),
					'choose_from_most_used'      => __( 'Choose from the most used writers', 'netbase-toolkit' ),
					'not_found'                  => __( 'No writers found.', 'netbase-toolkit' ),
					'menu_name'                  => __( 'Tags', 'netbase-toolkit' ),
				),
				'hierarchical'          => false,
				'show_ui'               => true,
				'show_admin_column'     => true,
				'update_count_callback' => '_update_post_term_count',
				'query_var'             => true,
				'rewrite'               => array( 'slug' => 'toolkit_tag' ),
			)
		);
	}

	/**
	 * Toolkit update messages.
	 *
	 * See /wp-admin/edit-form-advanced.php
	 *
	 * @param array $messages Existing post update messages.
	 *
	 * @return array Amended post update messages with new CPT update messages.
	 */
	public static function updated_messages( $messages ) {
		$post             = get_post();
		$post_type        = get_post_type( $post );

		if( $post_type == 'netbase-toolkit' ) {
			$post_type_object = get_post_type_object( $post_type );

			$messages['netbase-toolkit'] = array(
				0  => '', // Unused. Messages start at index 1.
				1  => __( 'Toolkit updated.', 'netbase-toolkit' ),
				2  => __( 'Custom field updated.', 'netbase-toolkit' ),
				3  => __( 'Custom field deleted.', 'netbase-toolkit' ),
				4  => __( 'Toolkit updated.', 'netbase-toolkit' ),
				/* translators: %s: date and time of the revision */
				5  => isset( $_GET['revision'] ) ? sprintf( __( 'Toolkit restored to revision from %s', 'netbase-toolkit' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
				6  => __( 'Toolkit published.', 'netbase-toolkit' ),
				7  => __( 'Toolkit saved.', 'netbase-toolkit' ),
				8  => __( 'Toolkit submitted.', 'netbase-toolkit' ),
				9  => sprintf(
					__( 'Toolkit scheduled for: <strong>%1$s</strong>.', 'netbase-toolkit' ),
					// translators: Publish box date format, see http://php.net/date
					date_i18n( __( 'M j, Y @ G:i', 'netbase-toolkit' ), strtotime( $post->post_date ) )
				),
				10 => __( 'Toolkit draft updated.', 'netbase-toolkit' )
			);

			if ( $post_type_object->publicly_queryable ) {
				$permalink = get_permalink( $post->ID );

				$view_link = sprintf( ' <a href="%s">%s</a>', esc_url( $permalink ), __( 'View toolkit', 'netbase-toolkit' ) );
				$messages[ $post_type ][1] .= $view_link;
				$messages[ $post_type ][6] .= $view_link;
				$messages[ $post_type ][9] .= $view_link;

				$preview_permalink = add_query_arg( 'preview', 'true', $permalink );
				$preview_link = sprintf( ' <a target="_blank" href="%s">%s</a>', esc_url( $preview_permalink ), __( 'Preview toolkit', 'netbase-toolkit' ) );
				$messages[ $post_type ][8]  .= $preview_link;
				$messages[ $post_type ][10] .= $preview_link;
			}
		}

		return $messages;
	}

	/**
	 * Load single item template file for the toolkit custom post type.
	 *
	 * @param   string  $template  Current template file.
	 *
	 * @return  string
	 */
	public static function single_template( $template ) {
		global $post;

		if ( isset( $post->post_type ) && $post->post_type == 'netbase-toolkit' ) {
			$template = NETBASE_TOOLKIT_PATH . 'templates/single.php';
		}

		return $template;
	}

	/**
	 * Load archive template file for the toolkit custom post type.
	 *
	 * @param   string  $template  Current template file.
	 *
	 * @return  string
	 */
	public static function archive_template( $template ) {
		global $post;


        $tax_file = NETBASE_TOOLKIT_PATH . 'templates/archive.php';
		if ( isset( $post->post_type ) && $post->post_type == 'netbase-toolkit' ) {
			$template = $tax_file;
		}

        if ( is_tax( 'toolkit_cat' ) ) {
            $template = $tax_file;
        }

		return $template;
	}

	/**
	 * Define helper function to print related toolkit.
	 *
	 * @return  array
	 */
	public static function related() {
		global $post;

		// Get the toolkit tags.
		$cats = get_the_terms( $post, 'toolkit_cat' );

		if ( ! class_exists( 'NB_Netbase' ) ) return;
		$nb_netbase_options = NB_Netbase::get_options();

		if ( $cats ) {
			$cat_ids = array();

			foreach ( $cats as $cat ) {
				$cat_ids[] = $cat->term_id;
			}

			$args = array(
				'post_type'      => 'netbase-toolkit',
				'post__not_in'   => array( $post->ID ),
				'posts_per_page' => 12,
				'tax_query'      => array(
					array(
						'taxonomy' => 'toolkit_cat',
						'field'    => 'id',
						'terms'    => $cat_ids,
					),
				)
			);

			$the_query = new WP_Query( $args );
			?>
			<div class="related-toolkit grid w-info mgt70">
				<h4 class="widget-title"><?php _e( 'Related Galleries', 'netbase-toolkit' ); ?></h4>

				<div class="related clear nb-netbase-carousel" data-owl-options='{"items": "4", "tablet": "3", "mobile": "1"<?php echo ( $nb_netbase_options['rtl'] ? ',"rtl": "true"' : '' ); ?>}'>
					<?php while ( $the_query->have_posts() ) : $the_query->the_post(); ?>
					<figure class="hentry">
						<div class="mask pr">
							<?php
								if ( has_post_thumbnail() ) :

									$img = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), '405x300' );

									echo '<a href="' . esc_url( get_permalink() ) . '">';
										echo '<img width="405" height="300" src="' . esc_url( $img[0] ) . '" alt="' . get_the_title() . '">';
									echo '</a>';

								else :
									echo '<img width="405" height="300" src="' . NETBASE_TOOLKIT_URL . 'assets/img/placeholder.png" alt="' . get_the_title() . '">';
								endif;
							?>
						</div><!-- .mask -->
						<figcaption>
							<div class="title mgt20">
								<h5><a href="<?php esc_url( the_permalink() ); ?>" title="<?php the_title(); ?>"><?php the_title(); ?></a></h5>
							</div><!-- .title -->
						</figcaption>
					</figure><!-- .hentry -->
					<?php endwhile; ?>
				</div><!-- .related -->
			</div><!-- .related-toolkit -->
		<?php
		}

		wp_reset_postdata();
	}

	/**
	 * Get excerpt.
	 *
	 * @param   integer  $limit        The number of words for an excerpt.
	 * @param   string   $after_limit  Read more text.
	 *
	 * @return  string
	 */
	public static function get_excerpt( $limit, $after_limit = '[...]' ) {
		$excerpt = get_the_excerpt();

		if ( $excerpt != '' ) {
			$excerpt = explode( ' ', strip_tags( strip_shortcodes( $excerpt ) ), $limit );
		} else {
			$excerpt = explode( ' ', strip_tags( strip_shortcodes( get_the_content() ) ), $limit );
		}

		if ( count( $excerpt ) < $limit ) {
			$excerpt = implode( ' ', $excerpt );
		} else {
			array_pop( $excerpt );

			$excerpt = implode( ' ', $excerpt ) . ' ' . $after_limit;
		}

		return $excerpt;
	}

	/**
	 * Get image links for creating toolkit.
	 *
	 * @param   string/array  $size  Image size.
	 *
	 * @return  array
	 */
	public static function get_multiple_image( $size = 'full' ) {
		$images = get_post_meta( get_the_ID(), 'multiple_image', false );

		$output = array();

		foreach ( $images as $key => $id ) {
			$link = wp_get_attachment_image_src( $id, $size );

			$output[$key][] = $link[0];
			$output[$key][] = $link[1];
			$output[$key][] = $link[2];
		}

		return $output;
	}

	/**
	 * Enqueue scripts and styles.
	 *
	 * @return  void
	 */
	public static function enqueue_scripts() {
		$layout     = get_theme_mod( 'toolkit_single_layout', 'grid' );
		$mousewheel = get_theme_mod( 'toolkit_single_mousewheel', 0 );

		if ( is_singular( 'netbase-toolkit' ) &&  'horizontal' == $layout && $mousewheel ) {
			wp_enqueue_script( 'mousewheel', NETBASE_TOOLKIT_URL . 'assets/vendors/mousewheel/jquery.mousewheel.min.js', array(), false, true  );
		}
	}
}
