"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var child_process_1 = __importDefault(require("child_process"));
/**
 * Converts supplied yml files to cli arguments
 * https://docs.docker.com/compose/reference/overview/#use--f-to-specify-name-and-path-of-one-or-more-compose-files
 */
var configToArgs = function (config) {
    if (typeof config === 'undefined') {
        return [];
    }
    else if (typeof config === 'string') {
        return ['-f', config];
    }
    else if (config instanceof Array) {
        return config.reduce(function (args, item) { return args.concat(['-f', item]); }, []);
    }
    throw new Error("Invalid argument supplied: " + config);
};
/**
 * Converts docker-compose commandline options to cli arguments
 */
var composeOptionsToArgs = function (composeOptions) {
    var composeArgs = [];
    composeOptions.forEach(function (option) {
        if (option instanceof Array) {
            composeArgs = composeArgs.concat(option);
        }
        if (typeof option === 'string') {
            composeArgs = composeArgs.concat([option]);
        }
    });
    return composeArgs;
};
/**
 * Executes docker-compose command with common options
 */
var execCompose = function (command, args, options) {
    if (options === void 0) { options = {}; }
    return new Promise(function (resolve, reject) {
        var composeOptions = options.composeOptions || [];
        var commandOptions = options.commandOptions || [];
        var composeArgs = composeOptionsToArgs(composeOptions);
        composeArgs = composeArgs.concat(configToArgs(options.config).concat([command].concat(composeOptionsToArgs(commandOptions), args)));
        var cwd = options.cwd;
        var env = options.env || undefined;
        var childProc = child_process_1.default.spawn('docker-compose', composeArgs, { cwd: cwd, env: env });
        childProc.on('error', function (err) {
            reject(err);
        });
        var result = {
            exitCode: null,
            err: '',
            out: ''
        };
        childProc.stdout.on('data', function (chunk) {
            result.out += chunk.toString();
        });
        childProc.stderr.on('data', function (chunk) {
            result.err += chunk.toString();
        });
        childProc.on('exit', function (exitCode) {
            result.exitCode = exitCode;
            if (exitCode === 0) {
                resolve(result);
            }
            else {
                reject(result);
            }
        });
        if (options.log) {
            childProc.stdout.pipe(process.stdout);
            childProc.stderr.pipe(process.stderr);
        }
    });
};
exports.upAll = function (options) {
    return execCompose('up', ['-d'], options);
};
exports.upMany = function (services, options) {
    return execCompose('up', ['-d'].concat(services), options);
};
exports.upOne = function (service, options) {
    return execCompose('up', ['-d', service], options);
};
exports.down = function (options) {
    return execCompose('down', [], options);
};
exports.stop = function (options) {
    return execCompose('stop', [], options);
};
exports.stopOne = function (service, options) {
    return execCompose('stop', [service], options);
};
exports.kill = function (options) {
    return execCompose('kill', [], options);
};
exports.rm = function (options) {
    return execCompose('rm', ['-f'], options);
};
exports.exec = function (container, command, options) {
    var args = Array.isArray(command) ? command : command.split(/\s+/);
    return execCompose('exec', ['-T', container].concat(args), options);
};
exports.run = function (container, command, options) {
    var args = Array.isArray(command) ? command : command.split(/\s+/);
    return execCompose('run', ['-T', container].concat(args), options);
};
exports.buildAll = function (options) {
    if (options === void 0) { options = {}; }
    return execCompose('build', options.parallel ? ['--parallel'] : [], options);
};
exports.buildMany = function (services, options) {
    if (options === void 0) { options = {}; }
    return execCompose('build', options.parallel ? ['--parallel'].concat(services) : services, options);
};
exports.buildOne = function (service, options) {
    return execCompose('build', [service], options);
};
exports.pullAll = function (options) {
    if (options === void 0) { options = {}; }
    return execCompose('pull', [], options);
};
exports.pullMany = function (services, options) {
    if (options === void 0) { options = {}; }
    return execCompose('pull', services, options);
};
exports.pullOne = function (service, options) {
    return execCompose('pull', [service], options);
};
exports.config = function (options) {
    return execCompose('config', [], options);
};
exports.configServices = function (options) {
    return execCompose('config', ['--services'], options);
};
exports.configVolumes = function (options) {
    return execCompose('config', ['--volumes'], options);
};
exports.ps = function (options) {
    return execCompose('ps', [], options);
};
exports.push = function (options) {
    if (options === void 0) { options = {}; }
    return execCompose('push', options.ignorePushFailures ? ['--ignore-push-failures'] : [], options);
};
exports.restartAll = function (options) {
    return execCompose('restart', [], options);
};
exports.restartMany = function (services, options) {
    return execCompose('restart', services, options);
};
exports.restartOne = function (service, options) {
    return exports.restartMany([service], options);
};
exports.logs = function (services, options) {
    if (options === void 0) { options = {}; }
    var args = Array.isArray(services) ? services : [services];
    if (options.follow) {
        args = ['--follow'].concat(args);
    }
    return execCompose('logs', args, options);
};
exports.port = function (service, containerPort, options) {
    var args = [service, containerPort];
    return execCompose('port', args, options);
};
