"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deactivatePlugin = exports.activatePlugin = exports.getPluginsMap = void 0;
/**
 * External dependencies
 */
const change_case_1 = require("change-case");
/**
 * Fetch the plugins from API and cache them in memory,
 * since they are unlikely to change during testing.
 *
 * @param this
 * @param forceRefetch Force refetch the installed plugins to update the cache.
 */
async function getPluginsMap(forceRefetch = false) {
    if (!forceRefetch && this.pluginsMap) {
        return this.pluginsMap;
    }
    const plugins = await this.rest({
        path: '/wp/v2/plugins',
    });
    this.pluginsMap = {};
    for (const plugin of plugins) {
        // Ideally, we should be using sanitize_title() in PHP rather than kebabCase(),
        // but we don't have the exact port of it in JS.
        // This is a good approximation though.
        const slug = (0, change_case_1.paramCase)(plugin.name.toLowerCase());
        this.pluginsMap[slug] = plugin.plugin;
    }
    return this.pluginsMap;
}
exports.getPluginsMap = getPluginsMap;
/**
 * Finds a plugin in the plugin map.
 *
 * Attempts to provide a helpful error message if not found.
 *
 * @param slug       Plugin slug.
 * @param pluginsMap Plugins map.
 */
function getPluginFromMap(slug, pluginsMap) {
    const plugin = pluginsMap[slug];
    if (!plugin) {
        for (const key of Object.keys(pluginsMap)) {
            if (key.toLowerCase().replace(/-/g, '') ===
                slug.toLowerCase().replace(/-/g, '')) {
                throw new Error(`The plugin "${slug}" isn't installed. Did you perhaps mean "${key}"?`);
            }
        }
        throw new Error(`The plugin "${slug}" isn't installed`);
    }
    return plugin;
}
/**
 * Activates an installed plugin.
 *
 * @param this RequestUtils.
 * @param slug Plugin slug.
 */
async function activatePlugin(slug) {
    const pluginsMap = await this.getPluginsMap();
    const plugin = getPluginFromMap(slug, pluginsMap);
    await this.rest({
        method: 'PUT',
        path: `/wp/v2/plugins/${plugin}`,
        data: { status: 'active' },
    });
}
exports.activatePlugin = activatePlugin;
/**
 * Deactivates an active plugin.
 *
 * @param this RequestUtils.
 * @param slug Plugin slug.
 */
async function deactivatePlugin(slug) {
    const pluginsMap = await this.getPluginsMap();
    const plugin = getPluginFromMap(slug, pluginsMap);
    await this.rest({
        method: 'PUT',
        path: `/wp/v2/plugins/${plugin}`,
        data: { status: 'inactive' },
    });
}
exports.deactivatePlugin = deactivatePlugin;
//# sourceMappingURL=plugins.js.map