"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.pressKeys = exports.setClipboardData = void 0;
/**
 * External dependencies
 */
const change_case_1 = require("change-case");
/**
 * WordPress dependencies
 */
const keycodes_1 = require("@wordpress/keycodes");
let clipboardDataHolder = {
    'text/plain': '',
    'text/html': '',
    'rich-text': '',
};
/**
 * Sets the clipboard data that can be pasted with
 * `pressKeys( 'primary+v' )`.
 *
 * @param this
 * @param clipboardData
 * @param clipboardData.plainText
 * @param clipboardData.html
 */
function setClipboardData({ plainText = '', html = '' }) {
    clipboardDataHolder = {
        'text/plain': plainText,
        'text/html': html,
        'rich-text': '',
    };
}
exports.setClipboardData = setClipboardData;
async function emulateClipboard(page, type) {
    clipboardDataHolder = await page.evaluate(([_type, _clipboardData]) => {
        const canvasDoc = 
        // @ts-ignore
        document.activeElement?.contentDocument ?? document;
        const clipboardDataTransfer = new DataTransfer();
        if (_type === 'paste') {
            clipboardDataTransfer.setData('text/plain', _clipboardData['text/plain']);
            clipboardDataTransfer.setData('text/html', _clipboardData['text/html']);
            clipboardDataTransfer.setData('rich-text', _clipboardData['rich-text']);
        }
        else {
            const selection = canvasDoc.defaultView.getSelection();
            const plainText = selection.toString();
            let html = plainText;
            if (selection.rangeCount) {
                const range = selection.getRangeAt(0);
                const fragment = range.cloneContents();
                html = Array.from(fragment.childNodes)
                    .map((node) => node.outerHTML ??
                    node.nodeValue)
                    .join('');
            }
            clipboardDataTransfer.setData('text/plain', plainText);
            clipboardDataTransfer.setData('text/html', html);
        }
        canvasDoc.activeElement?.dispatchEvent(new ClipboardEvent(_type, {
            bubbles: true,
            cancelable: true,
            clipboardData: clipboardDataTransfer,
        }));
        return {
            'text/plain': clipboardDataTransfer.getData('text/plain'),
            'text/html': clipboardDataTransfer.getData('text/html'),
            'rich-text': clipboardDataTransfer.getData('rich-text'),
        };
    }, [type, clipboardDataHolder]);
}
const isAppleOS = () => process.platform === 'darwin';
const isWebkit = (page) => page.context().browser().browserType().name() === 'webkit';
const browserCache = new WeakMap();
const getHasNaturalTabNavigation = async (page) => {
    if (!isAppleOS() || !isWebkit(page)) {
        return true;
    }
    if (browserCache.has(page.context().browser())) {
        return browserCache.get(page.context().browser());
    }
    const testPage = await page.context().newPage();
    await testPage.setContent(`<button>1</button><button>2</button>`);
    await testPage.getByText('1').focus();
    await testPage.keyboard.press('Tab');
    const featureDetected = await testPage
        .getByText('2')
        .evaluate((node) => node === document.activeElement);
    browserCache.set(page.context().browser(), featureDetected);
    await testPage.close();
    return featureDetected;
};
const modifiers = {
    ...keycodes_1.modifiers,
    shiftAlt: (_isApple) => _isApple() ? [keycodes_1.SHIFT, keycodes_1.ALT] : [keycodes_1.SHIFT, keycodes_1.CTRL],
};
async function pressKeys(key, { times, ...pressOptions } = {}) {
    const hasNaturalTabNavigation = await getHasNaturalTabNavigation(this.page);
    let command;
    if (key.toLowerCase() === 'primary+c') {
        command = () => emulateClipboard(this.page, 'copy');
    }
    else if (key.toLowerCase() === 'primary+x') {
        command = () => emulateClipboard(this.page, 'cut');
    }
    else if (key.toLowerCase() === 'primary+v') {
        command = () => emulateClipboard(this.page, 'paste');
    }
    else {
        const keys = key.split('+').flatMap((keyCode) => {
            if (Object.prototype.hasOwnProperty.call(modifiers, keyCode)) {
                return modifiers[keyCode](isAppleOS).map((modifier) => modifier === keycodes_1.CTRL ? 'Control' : (0, change_case_1.capitalCase)(modifier));
            }
            else if (keyCode === 'Tab' && !hasNaturalTabNavigation) {
                return ['Alt', 'Tab'];
            }
            return keyCode;
        });
        const normalizedKeys = keys.join('+');
        command = () => this.page.keyboard.press(normalizedKeys);
    }
    times = times ?? 1;
    for (let i = 0; i < times; i += 1) {
        await command();
        if (times > 1 && pressOptions.delay) {
            await this.page.waitForTimeout(pressOptions.delay);
        }
    }
}
exports.pressKeys = pressKeys;
//# sourceMappingURL=press-keys.js.map